<?php
/**
 * API Handler - TutorKita System
 * Handle all AJAX requests and API endpoints
 */

// Set maximum execution time to 30 seconds
ini_set('max_execution_time', 30);
ini_set('memory_limit', '128M');

// Start session for authentication
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Set session cookie parameters for better reliability
if (session_status() === PHP_SESSION_ACTIVE) {
    $sessionCookieParams = session_get_cookie_params();
    setcookie(
        session_name(),
        session_id(),
        time() + 3600, // 1 hour
        $sessionCookieParams['path'],
        $sessionCookieParams['domain'],
        false, // secure
        true // httponly
    );
}

// Check if db.php exists
// Check if db.php exists - check removed to force error if missing
// Updated at 2026-01-31 to fix cache issues
require_once __DIR__ . '/db.php';


// Set content type to JSON
header('Content-Type: application/json');

// Enable CORS for development
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Get request action
$action = isset($_REQUEST['action']) ? $_REQUEST['action'] : '';

// API Router
try {
    // Proactive Schema Check (Ensure columns exist before any action)
    if (in_array($action, ['register_student', 'register_student_pending', 'register_student_with_assign', 'get_student_management_data', 'update_student', 'add_student_manual', 'toggle_maintenance_mode', 'update_admin_settings', 'save_footer_settings', 'change_admin_password', 'update_admin_status'])) {
        try {
            // Check students table columns
            $colsFetch = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_COLUMN);

            if (!in_array('enrolled_by', $colsFetch)) {
                $pdo->exec("ALTER TABLE students ADD COLUMN enrolled_by INT(11) DEFAULT NULL AFTER status");
            }
            if (!in_array('parent_guardian_name', $colsFetch)) {
                $pdo->exec("ALTER TABLE students ADD COLUMN parent_guardian_name VARCHAR(255) DEFAULT NULL");
            }
            if (!in_array('education_level', $colsFetch)) {
                $pdo->exec("ALTER TABLE students ADD COLUMN education_level VARCHAR(100) DEFAULT NULL");
            }
            if (!in_array('grade_level', $colsFetch)) {
                $pdo->exec("ALTER TABLE students ADD COLUMN grade_level VARCHAR(100) DEFAULT NULL");
            }

            // Check student_enrollments table
            try {
                $enrollCols = $pdo->query("SHOW COLUMNS FROM student_enrollments")->fetchAll(PDO::FETCH_COLUMN);
                if (!in_array('auto_assigned', $enrollCols)) {
                    $pdo->exec("ALTER TABLE student_enrollments ADD COLUMN auto_assigned ENUM('Yes','No') DEFAULT 'No'");
                }
            } catch (Exception $e2) { /* Table might not exist yet */
            }

            // Check admin table columns
            try {
                $adminCols = $pdo->query("SHOW COLUMNS FROM admin")->fetchAll(PDO::FETCH_COLUMN);
                $missingAdminCols = [
                    'profile_name' => "VARCHAR(255) DEFAULT 'Admin'",
                    'profile_emoji' => "VARCHAR(50) DEFAULT '👨‍💻'",
                    'portal_logo' => "VARCHAR(255) DEFAULT ''",
                    'portal_favicon' => "VARCHAR(255) DEFAULT ''",
                    'portal_background' => "VARCHAR(255) DEFAULT ''",
                    'theme_color' => "VARCHAR(20) DEFAULT '#667eea'",
                    'font_type' => "VARCHAR(50) DEFAULT 'Segoe UI'",
                    'font_size' => "VARCHAR(10) DEFAULT '14px'",
                    'footer_text' => "VARCHAR(255) DEFAULT '© 2026 Hak Cipta Terpelihara'",
                    'maintenance_mode' => "ENUM('0','1') DEFAULT '0'"
                ];

                foreach ($missingAdminCols as $col => $definition) {
                    if (!in_array($col, $adminCols)) {
                        $pdo->exec("ALTER TABLE admin ADD COLUMN $col $definition");
                    }
                }

                // Ensure at least one admin record exists
                $adminCount = $pdo->query("SELECT COUNT(*) FROM admin")->fetchColumn();
                if ($adminCount == 0) {
                    $pdo->exec("INSERT INTO admin (username, password, profile_name, status) VALUES ('admin', '" . password_hash('admin123', PASSWORD_DEFAULT) . "', 'Administrator', 'Active')");
                }
            } catch (Exception $eAdmin) {
                // Table might not exist or other DB error
            }

        } catch (Exception $e) { /* Ignore */
        }
    }

    // Check forums table for attachments
    try {
        $fCols = $pdo->query("SHOW COLUMNS FROM forums")->fetchAll(PDO::FETCH_COLUMN);
        if (!in_array('attachment', $fCols)) {
            $pdo->exec("ALTER TABLE forums ADD COLUMN attachment VARCHAR(255) DEFAULT NULL");
        }
        if (!in_array('attachment_type', $fCols)) {
            $pdo->exec("ALTER TABLE forums ADD COLUMN attachment_type VARCHAR(50) DEFAULT NULL");
        }
    } catch (Exception $eF) { /* Table might not exist yet */
    }

    // Check forum_replies table for attachments
    try {
        $frCols = $pdo->query("SHOW COLUMNS FROM forum_replies")->fetchAll(PDO::FETCH_COLUMN);
        if (!in_array('attachment', $frCols)) {
            $pdo->exec("ALTER TABLE forum_replies ADD COLUMN attachment VARCHAR(255) DEFAULT NULL");
        }
        if (!in_array('attachment_type', $frCols)) {
            $pdo->exec("ALTER TABLE forum_replies ADD COLUMN attachment_type VARCHAR(50) DEFAULT NULL");
        }
    } catch (Exception $eFr) { /* Table might not exist yet */
    }

    // Check schedule table for selected_students
    try {
        $schedCols = $pdo->query("SHOW COLUMNS FROM schedule")->fetchAll(PDO::FETCH_COLUMN);
        if (!in_array('selected_students', $schedCols)) {
            $pdo->exec("ALTER TABLE schedule ADD COLUMN selected_students TEXT DEFAULT NULL");
        }
    } catch (Exception $eSched) { /* Table might not exist yet */
    }


    switch ($action) {
        // ========== AUTHENTICATION ==========
        case 'login':
            handleLogin();
            break;
        case 'logout':
            handleLogout();
            break;
        case 'check_session':
            checkSession();
            break;

        // ========== ADMIN OPERATIONS ==========
        case 'admin_stats':
            getAdminStats();
            break;
        case 'get_all_teachers':
            getAllTeachers();
            break;
        case 'add_teacher':
            addTeacher();
            break;
        case 'delete_teacher':
            deleteTeacher();
            break;
        case 'update_teacher_status_by_admin':
            updateTeacherStatusByAdmin();
            break;
        case 'update_admin_settings':
            updateAdminSettings();
            break;
        case 'get_admin_settings':
            getAdminSettings();
            break;
        case 'change_admin_password':
            changeAdminPassword();
            break;
        case 'reset_teacher_password':
            resetTeacherPassword();
            break;
        case 'toggle_maintenance_mode':
            toggleMaintenanceMode();
            break;

        // ========== STATUS MANAGEMENT ==========
        case 'update_teacher_status':
            updateTeacherStatus();
            break;
        case 'update_admin_status':
            updateAdminStatus();
            break;

        // ========== TEACHER OPERATIONS ==========
        case 'get_teacher_stats':
            getTeacherStats();
            break;
        case 'get_teacher_students':
            getTeacherStudents();
            break;
        case 'add_student_manual':
            addStudentManual();
            break;
        case 'update_student':
            updateStudent();
            break;
        case 'delete_student':
            deleteStudent();
            break;
        case 'resign_mentor':
            resignMentor();
            break;
        case 'get_teacher_profile':
            getTeacherProfile();
            break;
        case 'update_teacher_profile':
            updateTeacherProfile();
            break;
        case 'change_teacher_password':
            changeTeacherPassword();
            break;
        case 'get_all_schedules':
            getAllSchedules();
            break;

        // ========== STUDENT OPERATIONS ==========

        case 'get_opr_settings':
            getOPRSettings();
            break;
        case 'save_opr_settings':
            saveOprSettings();
            break;
        case 'add_maintenance_log':
            addMaintenanceLog();
            break;
        case 'get_maintenance_logs':
            getMaintenanceLogs();
            break;
        case 'delete_maintenance_log':
            deleteMaintenanceLog();
            break;
        case 'update_maintenance_log':
            updateMaintenanceLog();
            break;
        case 'save_footer_settings':
            saveFooterSettings();
            break;

        case 'setup_student_password':
            setupStudentPassword();
            break;

        // ========== STUDENT OPERATIONS ==========
        case 'get_student_profile':
            getStudentProfile();
            break;
        case 'get_student_contacts':
            getStudentContacts();
            break;
        case 'update_student_profile':
            updateStudentProfile();
            break;
        case 'change_student_password':
            changeStudentPassword();
            break;
        case 'get_student_subjects':
            getStudentSubjects();
            break;

        // ========== SUBJECTS ==========
        case 'get_subjects':
            getSubjects();
            break;
        case 'get_subjects_by_level':
            getSubjectsByLevel();
            break;

        // ========== NOTIFICATIONS ==========
        case 'get_notifications':
            getNotifications();
            break;
        case 'mark_notification_read':
            markNotificationRead();
            break;
        case 'delete_notification':
            deleteNotification();
            break;
        case 'delete_all_notifications':
            deleteAllNotifications();
            break;

        // ========== ANNOUNCEMENTS ==========
        case 'get_announcements':
            getAnnouncements();
            break;
        case 'get_teacher_announcements':
            getTeacherAnnouncements();
            break;
        case 'get_admin_announcements':
            getAdminAnnouncements();
            break;
        case 'add_announcement':
            addAnnouncement();
            break;
        case 'add_announcement_by_admin':
            addAnnouncementByAdmin();
            break;
        case 'delete_announcement':
            deleteAnnouncement();
            break;
        case 'update_announcement':
            updateAnnouncement();
            break;
        case 'delete_announcement_by_admin':
            deleteAnnouncementByAdmin();
            break;
        case 'get_latest_announcement':
            getLatestAnnouncement();
            break;

        // ========== REGISTRATION ==========
        case 'register_student':
            registerStudent();
            break;
        case 'register_teacher':
            registerTeacher();
            break;
        case 'check_username':
            checkUsername();
            break;
        case 'register_student_pending':
            registerStudentPending();
            break;

        // ========== MODULE FUNCTIONS ==========
        case 'get_teacher_modules':
            getTeacherModules();
            break;
        case 'get_module_students':
            getModuleStudents();
            break;
        case 'add_module':
            addModule();
            break;
        case 'update_module':
            updateModule();
            break;
        case 'delete_module':
            deleteModule();
            break;
        case 'get_student_modules':
            getStudentModules();
            break;
        case 'enroll_student_module':
            enrollStudentModule();
            break;

        // ========== SCHEDULE FUNCTIONS ==========
        case 'get_teacher_schedule':
            getTeacherSchedule();
            break;
        case 'add_schedule':
            addSchedule();
            break;
        case 'update_schedule':
            updateSchedule();
            break;
        case 'delete_schedule':
            deleteSchedule();
            break;
        case 'get_student_schedule':
            getStudentSchedule();
            break;

        // ========== ONLINE CLASS FUNCTIONS ==========
        case 'get_teacher_classes':
            getTeacherClasses();
            break;
        case 'add_class':
            addClass();
            break;
        case 'update_class':
            updateClass();
            break;
        case 'delete_class':
            deleteClass();
            break;
        case 'get_student_classes':
            getStudentClasses();
            break;
        case 'enroll_student_class':
            enrollStudentClass();
            break;
        case 'class_comment':
            classComment();
            break;

        // ========== AUTO-ASSIGNMENT FUNCTIONS ==========
        case 'register_student_with_assign':
            registerStudentWithAssign();
            break;
        case 'get_teacher_enrollments':
            getTeacherEnrollments();
            break;
        case 'approve_student':
            approveStudent();
            break;
        case 'reject_student':
            rejectStudent();
            break;
        case 'update_enrollment':
            updateEnrollment();
            break;
        case 'get_student_enrollments':
            getStudentEnrollments();
            break;
        case 'teacher_register_student':
            teacherRegisterStudent();
            break;
        case 'get_available_teachers':
            getAvailableTeachers();
            break;
        case 'get_subject_requests':
            getSubjectRequests();
            break;

        // ========== OPEN POOL LOGIC SUPPORT ==========
        case 'get_open_pool_students':
            getOpenPoolStudents();
            break;
        case 'approve_open_pool_student':
            approveOpenPoolStudent();
            break;
        case 'approve_student_with_password':
            approveStudentWithPassword();
            break;
        case 'sync_all_teacher_students':
            syncAllTeacherStudents();
            break;
        case 'reject_student_with_fallback':
            rejectStudentWithFallback();
            break;
        case 'get_available_teachers_by_subject':
            getAvailableTeachersBySubject();
            break;
        case 'manual_assign_student':
            manualAssignStudent();
            break;
        case 'get_pending_password_students':
            getPendingPasswordStudents();
            break;

        case 'reject_with_auto_fallback':
            rejectWithAutoFallback();
            break;

        // ========== FORUM FUNCTIONS ==========
        case 'get_teacher_forums':
            getTeacherForums();
            break;
        case 'add_forum':
            addForum();
            break;
        case 'update_forum':
            updateForum();
            break;
        case 'delete_forum':
            deleteForum();
            break;
        case 'get_student_forums':
            getStudentForums();
            break;
        case 'add_forum_reply':
            addForumReply();
            break;
        case 'get_forum_replies':
            getForumReplies();
            break;

        // ========== ASSIGNMENT/TUGASAN FUNCTIONS ==========
        case 'add_assignment':
            addAssignment();
            break;
        case 'update_assignment':
            updateAssignment();
            break;
        case 'delete_assignment':
            deleteAssignment();
            break;
        case 'get_teacher_assignments':
            getTeacherAssignments();
            break;
        case 'get_student_assignments':
            getStudentAssignments();
            break;
        case 'add_assignment_comment':
            addAssignmentComment();
            break;
        case 'get_assignment_comments':
            getAssignmentComments();
            break;

        // ========== DEBUG FUNCTIONS ==========
        case 'debug_teacher':
            // Debug function to check teacher login
            $username = isset($_POST['username']) ? trim($_POST['username']) : '';
            $password = isset($_POST['password']) ? $_POST['password'] : '';

            try {
                $stmt = $pdo->prepare("SELECT * FROM teachers WHERE username = ?");
                $stmt->execute([$username]);
                $teacher = $stmt->fetch();

                if (!$teacher) {
                    echo json_encode(['success' => false, 'message' => 'Guru tidak dijumpai dengan username: ' . $username]);
                } else {
                    echo json_encode([
                        'success' => true,
                        'debug' => [
                            'id' => $teacher['id'],
                            'username' => $teacher['username'],
                            'full_name' => $teacher['full_name'],
                            'status' => $teacher['status'],
                            'password_hash' => substr($teacher['password'], 0, 10) . '...',
                            'password_verify_test' => password_verify($password, $teacher['password']),
                            'raw_password' => $password
                        ]
                    ]);
                }
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            break;

        // ========== STUDENT MANAGEMENT COMBINED DATA ==========
        case 'get_student_management_data':
            getStudentManagementData();
            break;

        case 'approve_student_application':
            approveStudentApplication();
            break;
        case 'assign_student_mentor':
            assignStudentMentor();
            break;

        // ========== BILIK TUTOR FUNCTIONS ==========
        case 'get_tutor_messages':
            getTutorMessages();
            break;
        case 'send_tutor_message':
            sendTutorMessage();
            break;
        case 'toggle_pin_tutor_message':
            togglePinTutorMessage();
            break;
        case 'get_tutor_files':
            getTutorFiles();
            break;
        case 'get_teacher_list':
            getTeacherList();
            break;
        case 'upload_tutor_file':
            uploadTutorFile();
            break;
        case 'update_tutor_file':
            updateTutorFile();
            break;
        case 'delete_tutor_file':
            deleteTutorFile();
            break;

        // ========== DEFAULT - MUST BE LAST ==========
        default:
            echo json_encode(['success' => false, 'message' => 'Tindakan tidak dikenalpasti']);
            break;

    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

// ========== AUTHENTICATION FUNCTIONS ==========

function handleLogin()
{
    global $pdo;

    $username = isset($_POST['username']) ? trim($_POST['username']) : '';
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $userType = isset($_POST['user_type']) ? $_POST['user_type'] : '';

    if (empty($username) || empty($password) || empty($userType)) {
        echo json_encode(['success' => false, 'message' => 'Sila masukkan nama pengguna dan kata laluan']);
        return;
    }

    // Check maintenance mode for non-admin users
    if ($userType !== 'admin') {
        try {
            // Check if maintenance_mode column exists and is active
            $stmt = $pdo->query("SHOW COLUMNS FROM admin LIKE 'maintenance_mode'");
            $columnExists = $stmt->fetch();

            if ($columnExists) {
                $stmt = $pdo->prepare("SELECT maintenance_mode FROM admin WHERE id = 1");
                $stmt->execute();
                $adminSettings = $stmt->fetch();

                if ($adminSettings && $adminSettings['maintenance_mode'] == '1') {
                    echo json_encode(['success' => false, 'message' => 'Sistem sedang dalam penyelenggaraan. Sila cuba lagi kemudian.']);
                    return;
                }
            }
        } catch (Exception $e) {
            // Continue with login if maintenance check fails (column doesn't exist yet)
        }
    }

    try {
        if ($userType === 'admin') {
            $stmt = $pdo->prepare("SELECT * FROM admin WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if ($user && password_verify($password, $user['password'])) {
                // Regenerate session ID for security
                session_regenerate_id(true);

                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['user_type'] = 'admin';
                $_SESSION['profile_name'] = $user['profile_name'];
                $_SESSION['profile_emoji'] = $user['profile_emoji'];
                $_SESSION['login_time'] = time();

                session_write_close();

                // Create notification
                createNotification($user['id'], 'admin', 'Selamat Datang', 'Selamat sejahtera, ' . $user['profile_name'] . '!', 'success');

                echo json_encode([
                    'success' => true,
                    'message' => 'Log masuk berjaya!',
                    'user' => [
                        'id' => $user['id'],
                        'username' => $user['username'],
                        'profile_name' => $user['profile_name'],
                        'profile_emoji' => $user['profile_emoji'],
                        'user_type' => 'admin'
                    ]
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Nama pengguna atau kata laluan salah']);
            }
        } elseif ($userType === 'teacher') {
            $stmt = $pdo->prepare("SELECT * FROM teachers WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if ($user && password_verify($password, $user['password'])) {
                if ($user['status'] !== 'Active') {
                    echo json_encode(['success' => false, 'message' => 'Akaun anda telah dinyahaktifkan. Sila hubungi admin.']);
                    return;
                }
                // Regenerate session ID for security
                session_regenerate_id(true);

                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['user_type'] = 'teacher';
                $_SESSION['full_name'] = $user['full_name'];
                $_SESSION['profile_emoji'] = $user['profile_emoji'];
                $_SESSION['login_time'] = time();

                session_write_close();

                createNotification($user['id'], 'teacher', 'Selamat Datang', 'Selamat sejahtera, ' . $user['full_name'] . '!', 'success');

                echo json_encode([
                    'success' => true,
                    'message' => 'Log masuk berjaya!',
                    'user' => [
                        'id' => $user['id'],
                        'username' => $user['username'],
                        'full_name' => $user['full_name'],
                        'profile_emoji' => $user['profile_emoji'],
                        'user_type' => 'teacher'
                    ]
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Nama pengguna atau kata laluan salah']);
            }
        } elseif ($userType === 'student') {
            // Allow login via username OR phone for students
            $stmt = $pdo->prepare("SELECT * FROM students WHERE (username = ? OR phone = ?) AND status = 'Active'");
            $stmt->execute([$username, $username]);
            $user = $stmt->fetch();

            if ($user && password_verify($password, $user['password'])) {
                // Regenerate session ID for security
                session_regenerate_id(true);

                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['user_type'] = 'student';
                $_SESSION['full_name'] = $user['full_name'];
                $_SESSION['profile_emoji'] = $user['profile_emoji'];
                $_SESSION['login_time'] = time();

                session_write_close();

                createNotification($user['id'], 'student', 'Selamat Datang', 'Selamat sejahtera, ' . $user['full_name'] . '!', 'success');

                echo json_encode([
                    'success' => true,
                    'message' => 'Log masuk berjaya!',
                    'user' => [
                        'id' => $user['id'],
                        'username' => $user['username'],
                        'full_name' => $user['full_name'],
                        'profile_emoji' => $user['profile_emoji'],
                        'user_type' => 'student'
                    ]
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Nama pengguna atau kata laluan salah']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Jenis pengguna tidak valid']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function handleLogout()
{
    session_destroy();
    echo json_encode(['success' => true, 'message' => 'Log keluar berjaya']);
}

function checkSession()
{
    if (isset($_SESSION['user_id'])) {
        echo json_encode([
            'success' => true,
            'logged_in' => true,
            'user' => [
                'id' => $_SESSION['user_id'],
                'username' => $_SESSION['username'],
                'user_type' => $_SESSION['user_type'],
                'profile_name' => $_SESSION['full_name'], // Force full name
                'profile_emoji' => $_SESSION['profile_emoji']
            ]
        ]);
    } else {
        echo json_encode(['success' => true, 'logged_in' => false]);
    }
}

// ========== ADMIN FUNCTIONS ==========

function requireAdmin()
{
    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        exit;
    }
}

function getAdminStats()
{
    global $pdo;
    requireAdmin();

    try {
        $teacherCount = $pdo->query("SELECT COUNT(*) FROM teachers WHERE status = 'Active'")->fetchColumn();
        $studentCount = $pdo->query("SELECT COUNT(*) FROM students WHERE status = 'Active'")->fetchColumn();
        $subjectCount = $pdo->query("SELECT COUNT(*) FROM subjects WHERE is_active = 'Yes'")->fetchColumn();

        echo json_encode([
            'success' => true,
            'stats' => [
                'teachers' => $teacherCount,
                'students' => $studentCount,
                'subjects' => $subjectCount
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getAllTeachers()
{
    global $pdo;
    requireAdmin();

    try {
        $stmt = $pdo->query("SELECT id, username, full_name, phone, education_level, grade_level, subjects, status, created_at FROM teachers ORDER BY id DESC");
        $teachers = $stmt->fetchAll();

        echo json_encode(['success' => true, 'teachers' => $teachers]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addTeacher()
{
    global $pdo;

    // Pastikan user adalah admin
    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
        echo json_encode(['success' => false, 'message' => 'Akses dinafikan']);
        return;
    }

    // Ambil data dari POST
    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';

    // --- PEMBETULAN DI SINI ---
    // JavaScript hantar sebagai 'grades[]', jadi PHP terima sebagai $_POST['grades']
    $gradesArray = isset($_POST['grades']) ? $_POST['grades'] : [];
    $subjectsArray = isset($_POST['subjects']) ? $_POST['subjects'] : [];

    // Validasi: Pastikan field penting diisi
    if (empty($fullName) || empty($phone) || empty($educationLevel) || empty($subjectsArray)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi Nama, Telefon, Tahap dan Subjek.']);
        return;
    }

    // Tukar array kepada JSON string supaya SQL boleh simpan dalam kolum TEXT
    $gradeLevelJSON = json_encode($gradesArray);
    $subjectsJSON = json_encode($subjectsArray);

    // Generate username
    $username = strtolower(str_replace(' ', '', $fullName)) . rand(10, 99);
    $password = password_hash('tutor123', PASSWORD_DEFAULT);

    try {
        // Prepare SQL
        $stmt = $pdo->prepare("INSERT INTO teachers 
            (username, password, full_name, phone, education_level, grade_level, subjects, created_by) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)");

        $stmt->execute([
            $username,
            $password,
            $fullName,
            $phone,
            $educationLevel,
            $gradeLevelJSON, // Masukkan JSON string yang sudah diproses
            $subjectsJSON,   // Masukkan JSON string subjek
            $_SESSION['user_id']
        ]);

        echo json_encode([
            'success' => true,
            'message' => 'Guru berjaya ditambah!',
            'username' => $username,
            'password' => 'tutor123'
        ]);

    } catch (PDOException $e) {
        if ($e->getCode() == 23000) {
            echo json_encode(['success' => false, 'message' => 'Username atau No. Telefon sudah wujud.']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Ralat Database: ' . $e->getMessage()]);
        }
    }
}


function deleteTeacher()
{
    global $pdo;
    requireAdmin();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID guru tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE teachers SET status = 'Inactive' WHERE id = ?");
        $stmt->execute([$id]);

        echo json_encode(['success' => true, 'message' => 'Guru berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getAdminSettings()
{
    global $pdo;
    requireAdmin();

    try {
        $stmt = $pdo->prepare("SELECT * FROM admin WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $admin = $stmt->fetch();

        echo json_encode(['success' => true, 'settings' => $admin]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
function updateAdminSettings()
{
    global $pdo;
    requireAdmin();

    // 1. Ambil data profil & password
    $profileName = isset($_POST['profile_name']) ? trim($_POST['profile_name']) : '';
    $profileEmoji = isset($_POST['profile_emoji']) ? $_POST['profile_emoji'] : '👨‍💻';
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : ''; // Ambil password baru

    // 2. Ambil data portal
    $portalName = isset($_POST['portal_name']) ? trim($_POST['portal_name']) : '';
    $portalTagline = isset($_POST['portal_tagline']) ? trim($_POST['portal_tagline']) : '';
    $portalLogo = isset($_POST['portal_logo']) ? trim($_POST['portal_logo']) : '';
    $portalFavicon = isset($_POST['portal_favicon']) ? trim($_POST['portal_favicon']) : '';
    $portalBackground = isset($_POST['portal_background']) ? trim($_POST['portal_background']) : '';
    $themeColor = isset($_POST['theme_color']) ? $_POST['theme_color'] : '';
    $fontType = isset($_POST['font_type']) ? $_POST['font_type'] : 'Segoe UI';
    $fontSize = isset($_POST['font_size']) ? $_POST['font_size'] : '14px';
    $footerText = isset($_POST['footer_text']) ? $_POST['footer_text'] : ''; // Tambah footer text

    // 3. Ambil data maintenance mode
    $maintenanceMode = isset($_POST['maintenance_mode']) ? '1' : '0';

    try {
        // 4. Kemaskini maklumat umum & profil (termasuk footer_text, portal_background, dan maintenance_mode)
        $stmt = $pdo->prepare("UPDATE admin SET 
            profile_name = ?, 
            profile_emoji = ?, 
            portal_name = ?, 
            portal_tagline = ?, 
            portal_logo = ?, 
            portal_favicon = ?, 
            portal_background = ?,
            theme_color = ?, 
            font_type = ?, 
            font_size = ?,
            footer_text = ?,
            maintenance_mode = ? 
            WHERE id = ?");

        $stmt->execute([
            $profileName,
            $profileEmoji,
            $portalName,
            $portalTagline,
            $portalLogo,
            $portalFavicon,
            $portalBackground,
            $themeColor,
            $fontType,
            $fontSize,
            $footerText,
            $maintenanceMode,
            $_SESSION['user_id']
        ]);

        // 5. Proses tukar kata laluan (Hanya jika diisi)
        if (!empty($newPassword)) {
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmtPass = $pdo->prepare("UPDATE admin SET password = ? WHERE id = ?");
            $stmtPass->execute([$hashedPassword, $_SESSION['user_id']]);
        }

        // 6. Kemaskini Session supaya paparan berubah serta-merta
        $_SESSION['profile_name'] = $profileName;
        $_SESSION['profile_emoji'] = $profileEmoji;

        echo json_encode(['success' => true, 'message' => 'Profil dan Tetapan berjaya dikemaskini!']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function toggleMaintenanceMode()
{
    global $pdo;
    requireAdmin();

    $enabled = isset($_POST['enabled']) ? $_POST['enabled'] : '0';
    $status = ($enabled === 'true' || $enabled === '1') ? '1' : '0';

    try {
        $stmt = $pdo->prepare("UPDATE admin SET maintenance_mode = ? WHERE id = 1");
        $stmt->execute([$status]);
        echo json_encode(['success' => true, 'message' => 'Mod penyelenggaraan berjaya dikemaskini!']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}


function changeAdminPassword()
{
    global $pdo;
    requireAdmin();

    $currentPassword = isset($_POST['current_password']) ? $_POST['current_password'] : '';
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';

    if ($newPassword !== $confirmPassword) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan baru tidak sepadan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT password FROM admin WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $admin = $stmt->fetch();

        if (!password_verify($currentPassword, $admin['password'])) {
            echo json_encode(['success' => false, 'message' => 'Kata laluan semasa salah']);
            return;
        }

        $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE admin SET password = ? WHERE id = ?");
        $stmt->execute([$newHash, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Kata laluan berjaya ditukar']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== TEACHER FUNCTIONS ==========

function requireTeacher()
{
    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        exit;
    }
}

function requireAdminOrTeacher()
{
    if (!isset($_SESSION['user_type']) || ($_SESSION['user_type'] !== 'admin' && $_SESSION['user_type'] !== 'teacher')) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        exit;
    }
}

function changeTeacherPassword()
{
    global $pdo;
    requireTeacher();

    $currentPassword = isset($_POST['current_password']) ? $_POST['current_password'] : '';
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';

    if ($newPassword !== $confirmPassword) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan baru tidak sepadan']);
        return;
    }

    if (strlen($newPassword) < 6) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan minima 6 aksara']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT password FROM teachers WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $teacher = $stmt->fetch();

        if (!$teacher || !password_verify($currentPassword, $teacher['password'])) {
            echo json_encode(['success' => false, 'message' => 'Kata laluan semasa salah']);
            return;
        }

        $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE teachers SET password = ? WHERE id = ?");
        $stmt->execute([$newHash, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Kata laluan berjaya ditukar']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
function getTeacherStats()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM students WHERE enrolled_by = ? AND status = 'Active'");
        $stmt->execute([$_SESSION['user_id']]);
        $studentCount = $stmt->fetchColumn();

        echo json_encode([
            'success' => true,
            'stats' => [
                'students' => $studentCount
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getTeacherStudents()
{
    global $pdo;
    requireTeacher();

    try {
        // Get teacher's subjects
        $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $teacher = $stmt->fetch();

        if (!$teacher) {
            echo json_encode(['success' => false, 'message' => 'Guru tidak dijumpai']);
            return;
        }

        // Parse teacher subjects
        $teacherSubjectsRaw = $teacher['subjects'] ?? '';
        $teacherSubjects = [];

        if (!empty($teacherSubjectsRaw)) {
            // Try to decode as JSON array
            $teacherSubjects = json_decode($teacherSubjectsRaw, true);

            // If JSON decode fails or result is not an array, try as comma-separated string
            if (json_last_error() !== JSON_ERROR_NONE || !is_array($teacherSubjects) || empty($teacherSubjects)) {
                // Try comma-separated format
                $teacherSubjects = array_map('trim', explode(',', $teacherSubjectsRaw));
            }
        }

        // Ensure teacherSubjects is always an array
        if (!is_array($teacherSubjects)) {
            $teacherSubjects = [];
        }

        if (empty($teacherSubjects)) {
            // Do not return early. Just proceed. 
            // We still want to show assigned students even if teacher has no subjects set.
        }

        // Query 1: Students where I am the Primary Mentor (enrolled_by)
        $queryPrimary = "
            SELECT s.* 
            FROM students s 
            WHERE s.status = 'Active' AND s.enrolled_by = ?
            ORDER BY s.created_at DESC
        ";
        $stmtP = $pdo->prepare($queryPrimary);
        $stmtP->execute([$_SESSION['user_id']]);
        $primaryStudents = $stmtP->fetchAll();

        // Query 2: Students where I am the Subject Teacher (student_enrollments)
        $querySecondary = "
            SELECT s.*, se.subject as assigned_subject
            FROM students s
            JOIN student_enrollments se ON s.id = se.student_id
            WHERE s.status = 'Active' AND se.teacher_id = ? AND se.status = 'Approved'
            ORDER BY s.created_at DESC
        ";
        $stmtS = $pdo->prepare($querySecondary);
        $stmtS->execute([$_SESSION['user_id']]);
        $secondaryStudents = $stmtS->fetchAll();

        // Merge students (Priority to Secondary for 'assigned_subject')
        // We us an associative array keyed by ID to merge
        $allStudentsMap = [];

        // First add Primary (they might not have assigned_subject set, so we set relation)
        foreach ($primaryStudents as $p) {
            $p['relation_type'] = 'Primary';
            $allStudentsMap[$p['id']] = $p;
        }

        // Then add/overwrite with Secondary (contains assigned_subject)
        foreach ($secondaryStudents as $s) {
            // If already exists (Primary), we just update the assigned_subject and relation info if needed.
            // But usually we want to preserve 'Primary' relation but ADD 'assigned_subject'.
            if (isset($allStudentsMap[$s['id']])) {
                $allStudentsMap[$s['id']]['assigned_subject'] = $s['assigned_subject'];
                // Keep relation_type as Primary if they are enrolled_by me, effectively they are both.
                // Or maybe just leave it provided the assigned_subject is set.
            } else {
                $s['relation_type'] = 'Secondary';
                $allStudentsMap[$s['id']] = $s;
            }
        }

        $students = array_values($allStudentsMap);

        // Group students by their enrolled subjects
        $studentsBySubject = [];

        foreach ($students as $student) {
            $isCategorized = false;

            // 1. Try to categorize by assigned_subject from enrollment (Priority)
            // Always use assigned_subject if it exists, don't skip General
            if (!empty($student['assigned_subject'])) {
                $subj = $student['assigned_subject'];
                if (!isset($studentsBySubject[$subj]))
                    $studentsBySubject[$subj] = [];

                $studentCopy = $student;
                $studentCopy['enrolled_subject'] = $subj;
                $studentsBySubject[$subj][] = $studentCopy;
                $isCategorized = true;
            }

            // 2. If not categorized, try matching required_subjects with teacher's subjects
            if (!$isCategorized) {
                $subjects = json_decode($student['required_subjects'] ?: '[]', true);
                if (is_array($subjects) && !empty($subjects)) {
                    foreach ($subjects as $subject) {
                        // If teacher has no specific subjects set, or matches one
                        if (empty($teacherSubjects) || in_array($subject, $teacherSubjects)) {
                            if (!isset($studentsBySubject[$subject])) {
                                $studentsBySubject[$subject] = [];
                            }
                            $studentCopy = $student;
                            $studentCopy['enrolled_subject'] = $subject;
                            $studentsBySubject[$subject][] = $studentCopy;
                            $isCategorized = true;
                        }
                    }
                }
            }

            // 3. If still not categorized (e.g. 'General' subject or no match), put in 'Lain-lain'
            if (!$isCategorized) {
                $subject = 'Lain-lain';
                if (!isset($studentsBySubject[$subject])) {
                    $studentsBySubject[$subject] = [];
                }
                // Use assigned_subject if available, else first required subject, else 'General'
                $displaySubject = $student['assigned_subject'] ?: (json_decode($student['required_subjects'] ?? '[]')[0] ?? 'General');

                $studentCopy = $student;
                $studentCopy['enrolled_subject'] = $displaySubject;
                $studentsBySubject[$subject][] = $studentCopy;
            }
        }

        // Create a flat list of processed students for the frontend
        $processedStudents = [];
        foreach ($studentsBySubject as $subj => $list) {
            foreach ($list as $s) {
                // Ensure enrolled_subject is set in the main object
                $s['enrolled_subject'] = $subj;
                $processedStudents[] = $s;
            }
        }

        // Remove duplicates if any (though grouping should prevent this, safety check)
        // Actually, listing by subject means if a student is in 2 subjects under same teacher, they appear twice?
        // Usually index-v2 expects unique students list.
        // Let's unique by ID, preferring the one with 'assigned_subject' match.

        $uniqueStudents = [];
        foreach ($processedStudents as $ps) {
            $id = $ps['id'];
            if (!isset($uniqueStudents[$id])) {
                $uniqueStudents[$id] = $ps;
            } else {
                // If existing is 'Lain-lain' and new is specific, overwrite
                if ($uniqueStudents[$id]['enrolled_subject'] === 'Lain-lain' && $ps['enrolled_subject'] !== 'Lain-lain') {
                    $uniqueStudents[$id] = $ps;
                }
            }
        }

        $finalList = array_values($uniqueStudents);

        // Kira statistik
        $activeCount = count($finalList);
        $pendingCount = 0; // Tiada enrollment, jadi tiada pending

        echo json_encode([
            'success' => true,
            'students' => $finalList, // Return the PROCESSED list
            'students_by_subject' => $studentsBySubject,
            'approved_students' => $finalList,
            'approved_count' => $activeCount,
            'pending_count' => $pendingCount,
            'message' => $activeCount > 0 ? 'Pelajar berjaya dimuatkan' : 'Tiada pelajar untuk subjek yang anda ajar'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}


function updateStudent()
{
    global $pdo;
    requireTeacher();

    $studentId = $_POST['student_id'] ?? ($_POST['id'] ?? (isset($_REQUEST['id']) ? $_REQUEST['id'] : 0));
    if ($studentId <= 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar tidak sah (' . $studentId . ')']);
        return;
    }

    // Check Mentor Authorization
    $stmtCheck = $pdo->prepare("SELECT enrolled_by FROM students WHERE id = ?");
    $stmtCheck->execute([$studentId]);
    $studentCheck = $stmtCheck->fetch();

    if (!$studentCheck || $studentCheck['enrolled_by'] != $_SESSION['user_id']) {
        echo json_encode(['success' => false, 'message' => 'Hanya mentor sahaja layak untuk kemaskini dan padam pelajar']);
        return;
    }

    $fullName = $_POST['full_name'] ?? '';
    $age = $_POST['age'] ?? '';
    $phone = $_POST['phone'] ?? '';
    $parentName = $_POST['parent_guardian_name'] ?? '';
    $educationLevel = $_POST['education_level'] ?? '';
    $gradeLevel = $_POST['grade_level'] ?? null;
    $password = $_POST['password'] ?? '';
    $username = $_POST['username'] ?? '';

    try {
        // Base Query
        $sql = "UPDATE students SET full_name=?, age=?, phone=?, parent_guardian_name=?";
        $params = [$fullName, $age, $phone, $parentName];

        if ($educationLevel) {
            $sql .= ", education_level=?";
            $params[] = $educationLevel;
        }

        if ($gradeLevel) {
            $sql .= ", grade_level=?";
            $params[] = $gradeLevel;
        }

        // Optional Username Update
        if (!empty($username)) {
            // Check uniqueness
            $stmtCheck = $pdo->prepare("SELECT id FROM students WHERE username=? AND id != ?");
            $stmtCheck->execute([$username, $studentId]);
            if ($stmtCheck->fetch()) {
                echo json_encode(['success' => false, 'message' => 'Username sudah digunakan oleh pelajar lain.']);
                return;
            }

            $sql .= ", username=?";
            $params[] = $username;
        }

        // Optional Password Update
        if (!empty($password)) {
            $sql .= ", password=?";
            $params[] = password_hash($password, PASSWORD_DEFAULT);
        }

        // Subjects Update
        $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : null;
        if ($requiredSubjects !== null) {
            // Encode if array, or ensure valid JSON if string
            if (is_array($requiredSubjects)) {
                $subjectsJson = json_encode($requiredSubjects);
            } else {
                $subjectsJson = $requiredSubjects; // Assume it's already a JSON string or handled
            }
            $sql .= ", required_subjects=?";
            $params[] = $subjectsJson;
        }

        $sql .= " WHERE id=?";
        $params[] = $studentId;

        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);

        if ($result) {
            echo json_encode(['success' => true, 'message' => 'Maklumat berjaya dikemaskini']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Gagal mengemaskini pangkalan data']);
        }

    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}

function deleteStudent()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar tidak valid']);
        return;
    }

    // Check Mentor Authorization
    $stmtCheck = $pdo->prepare("SELECT enrolled_by FROM students WHERE id = ?");
    $stmtCheck->execute([$id]);
    $studentCheck = $stmtCheck->fetch();

    if (!$studentCheck || $studentCheck['enrolled_by'] != $_SESSION['user_id']) {
        echo json_encode(['success' => false, 'message' => 'Hanya mentor sahaja layak untuk kemaskini dan padam pelajar']);
        return;
    }

    try {
        // Just deactivate student directly (no enrollment table needed)
        $stmt = $pdo->prepare("UPDATE students SET status = 'Inactive' WHERE id = ?");
        $stmt->execute([$id]);

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Resign as Mentor for a student
 * Sets enrolled_by to NULL so other teachers can take over
 */
function resignMentor()
{
    global $pdo;
    requireTeacher();

    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : (isset($_POST['id']) ? (int) $_POST['id'] : 0);

    if ($studentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar tidak sah']);
        return;
    }

    try {
        // Verify current user is the mentor
        $stmt = $pdo->prepare("SELECT enrolled_by FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if (!$student || $student['enrolled_by'] != $_SESSION['user_id']) {
            echo json_encode(['success' => false, 'message' => 'Anda bukan mentor utama pelajar ini.']);
            return;
        }

        // Set enrolled_by to NULL
        $stmt = $pdo->prepare("UPDATE students SET enrolled_by = NULL WHERE id = ?");
        $stmt->execute([$studentId]);

        echo json_encode(['success' => true, 'message' => 'Anda telah melepaskan tanggungjawab sebagai mentor pelajar ini.']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}

function getTeacherProfile()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->prepare("SELECT id, username, full_name, phone, education_level, grade_level, subjects, profile_emoji FROM teachers WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $teacher = $stmt->fetch();

        echo json_encode(['success' => true, 'profile' => $teacher]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateTeacherProfile()
{
    global $pdo;
    requireTeacher();

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $email = isset($_POST['email']) ? trim($_POST['email']) : '';
    $profileEmoji = isset($_POST['profile_emoji']) ? $_POST['profile_emoji'] : '';
    $subjects = isset($_POST['subjects']) ? $_POST['subjects'] : '';

    // Validate email format if provided
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Format email tidak sah']);
        return;
    }

    // Parse subjects JSON if it's a string
    if (is_string($subjects) && !empty($subjects)) {
        $subjectsArray = json_decode($subjects, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($subjectsArray)) {
            $subjects = json_encode($subjectsArray);
        } else {
            $subjects = '[]';
        }
    } elseif (is_array($subjects)) {
        $subjects = json_encode($subjects);
    } else {
        // Keep existing subjects if none provided
        $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $current = $stmt->fetch();
        $subjects = $current['subjects'] ?? '[]';
    }

    try {
        $stmt = $pdo->prepare("UPDATE teachers SET full_name = ?, phone = ?, email = ?, profile_emoji = ?, subjects = ? WHERE id = ?");
        $stmt->execute([$fullName, $phone, $email, $profileEmoji, $subjects, $_SESSION['user_id']]);

        $_SESSION['full_name'] = $fullName;
        $_SESSION['profile_emoji'] = $profileEmoji;

        echo json_encode(['success' => true, 'message' => 'Profil berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== STUDENT FUNCTIONS ==========

function requireStudent()
{
    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'student') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        exit;
    }
}

function getStudentProfile()
{
    global $pdo;

    // Allow both student (their own profile) and teacher to view student profiles
    if (!isset($_SESSION['user_type']) || !in_array($_SESSION['user_type'], ['student', 'teacher'])) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    // Students can only view their own profile, teachers can view any student
    $studentId = isset($_GET['student_id']) ? (int) $_GET['student_id'] : 0;

    if ($_SESSION['user_type'] === 'student' && $studentId === 0) {
        $studentId = $_SESSION['user_id'];
    }

    if ($studentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar diperlukan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("
            SELECT s.id, s.username, s.full_name, s.age, s.parent_guardian_name, s.phone, s.education_level, s.grade_level, s.required_subjects, s.status, s.created_at, s.enrolled_by,
                   t.full_name as mentor_name
            FROM students s
            LEFT JOIN teachers t ON s.enrolled_by = t.id
            WHERE s.id = ?
        ");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if (!$student) {
            echo json_encode(['success' => false, 'message' => 'Pelajar tidak dijumpai']);
            return;
        }

        echo json_encode(['success' => true, 'profile' => $student]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateStudentProfile()
{
    global $pdo;
    requireStudent();

    $profileEmoji = isset($_POST['profile_emoji']) ? $_POST['profile_emoji'] : '';

    try {
        // Only update emoji for now since that's what's in the form
        $stmt = $pdo->prepare("UPDATE students SET profile_emoji = ? WHERE id = ?");
        $stmt->execute([$profileEmoji, $_SESSION['user_id']]);

        $_SESSION['profile_emoji'] = $profileEmoji;

        echo json_encode(['success' => true, 'message' => 'Profil berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentSubjects()
{
    global $pdo;
    requireStudent();

    try {
        $stmt = $pdo->prepare("SELECT required_subjects FROM students WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $student = $stmt->fetch();

        echo json_encode(['success' => true, 'subjects' => json_decode($student['required_subjects'])]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentContacts()
{
    global $pdo;
    requireStudent();

    $studentId = $_SESSION['user_id'];

    try {
        $stmt = $pdo->prepare("
            SELECT t.id as mentor_id, t.full_name as mentor_name, t.phone as mentor_phone
            FROM students s
            LEFT JOIN teachers t ON s.enrolled_by = t.id
            WHERE s.id = ?
        ");
        $stmt->execute([$studentId]);
        $mentorRow = $stmt->fetch(PDO::FETCH_ASSOC);

        $mentor = null;
        $mentorId = null;
        if ($mentorRow && $mentorRow['mentor_id']) {
            $mentor = [
                'id' => (int)$mentorRow['mentor_id'],
                'full_name' => $mentorRow['mentor_name'],
                'phone' => $mentorRow['mentor_phone'],
            ];
            $mentorId = (int)$mentorRow['mentor_id'];
        }

        $stmt = $pdo->prepare("
            SELECT DISTINCT t.id, t.full_name, t.phone
            FROM student_enrollments se
            JOIN teachers t ON se.teacher_id = t.id
            WHERE se.student_id = ? AND se.status = 'Approved'
        ");
        $stmt->execute([$studentId]);
        $additional = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            if ($mentorId && (int)$row['id'] === $mentorId) continue;
            $additional[] = [
                'id' => (int)$row['id'],
                'full_name' => $row['full_name'],
                'phone' => $row['phone'],
            ];
        }

        echo json_encode(['success' => true, 'mentor' => $mentor, 'additional_teachers' => $additional]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
// ========== SUBJECT FUNCTIONS ==========

function getSubjects()
{
    global $pdo;

    try {
        $stmt = $pdo->query("SELECT * FROM subjects WHERE is_active = 'Yes' ORDER BY subject_name");
        $subjects = $stmt->fetchAll();

        echo json_encode(['success' => true, 'subjects' => $subjects]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getSubjectsByLevel()
{
    global $pdo;

    $level = isset($_GET['level']) ? $_GET['level'] : '';

    try {
        if ($level) {
            // Relaxed Query: Use LIKE to catch "Sekolah Menengah" matching "Sekolah Menengah Atas" etc if needed
            $stmt = $pdo->prepare("SELECT * FROM subjects WHERE is_active = 'Yes' AND (education_level LIKE ? OR education_level IS NULL OR education_level = '') ORDER BY subject_name");
            $stmt->execute(['%' . $level . '%']);
        } else {
            $stmt = $pdo->query("SELECT * FROM subjects WHERE is_active = 'Yes' ORDER BY subject_name");
        }
        $subjects = $stmt->fetchAll();

        echo json_encode(['success' => true, 'subjects' => $subjects]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== NOTIFICATION FUNCTIONS ==========

function getNotifications()
{
    global $pdo;

    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE user_id = ? AND user_type = ? ORDER BY created_at DESC LIMIT 20");
        $stmt->execute([$_SESSION['user_id'], $_SESSION['user_type']]);
        $notifications = $stmt->fetchAll();

        echo json_encode(['success' => true, 'notifications' => $notifications]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function markNotificationRead()
{
    global $pdo;

    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    try {
        if ($id > 0) {
            $stmt = $pdo->prepare("UPDATE notifications SET is_read = 'Yes' WHERE id = ? AND user_id = ?");
            $stmt->execute([$id, $_SESSION['user_id']]);
        } else {
            $stmt = $pdo->prepare("UPDATE notifications SET is_read = 'Yes' WHERE user_id = ? AND user_type = ?");
            $stmt->execute([$_SESSION['user_id'], $_SESSION['user_type']]);
        }

        echo json_encode(['success' => true, 'message' => 'Notifikasi dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteNotification()
{
    global $pdo;

    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID notifikasi tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("DELETE FROM notifications WHERE id = ? AND user_id = ? AND user_type = ?");
        $stmt->execute([$id, $_SESSION['user_id'], $_SESSION['user_type']]);

        if ($stmt->rowCount() > 0) {
            echo json_encode(['success' => true, 'message' => 'Notifikasi berjaya dipadam']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Notifikasi tidak dijumpai']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteAllNotifications()
{
    global $pdo;

    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    try {
        $stmt = $pdo->prepare("DELETE FROM notifications WHERE user_id = ? AND user_type = ?");
        $stmt->execute([$_SESSION['user_id'], $_SESSION['user_type']]);

        echo json_encode(['success' => true, 'message' => 'Semua notifikasi berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== ANNOUNCEMENT FUNCTIONS ==========

function getAnnouncements()
{
    global $pdo;

    if (!isset($_SESSION['user_type'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    // Ensure target_audience column exists
    try {
        $check = $pdo->query("SHOW COLUMNS FROM announcements LIKE 'target_audience'");
        if ($check->rowCount() == 0) {
            $pdo->query("ALTER TABLE announcements ADD COLUMN target_audience VARCHAR(20) DEFAULT 'Semua' AFTER priority");
        }
    } catch (Exception $e) { /* Ignore */
    }

    try {
        // Simplified query without student_enrollments
        // For teachers, show all announcements
        // For students, show announcements from teachers (simplified - no enrollment filter)

        $query = "
            SELECT a.*, COALESCE(t.full_name, 'Pentadbir') as teacher_name, t.profile_emoji
            FROM announcements a
            LEFT JOIN teachers t ON a.teacher_id = t.id
            WHERE (a.expiry_date IS NULL OR a.expiry_date >= CURDATE() OR a.expiry_date = '0000-00-00')
        ";

        // Filter based on user type
        if ($_SESSION['user_type'] === 'student') {
            $query .= " AND (a.target_audience IN ('Semua', 'Pelajar') OR a.target_audience IS NULL)";
        }
        // Admin sees all, Teacher has their own function but if they use this, show all relevant

        $query .= " ORDER BY 
                CASE WHEN a.priority = 'Urgent' THEN 1 
                     WHEN a.priority = 'Penting' THEN 2 
                     ELSE 3 END,
                a.created_at DESC";

        $stmt = $pdo->query($query);
        $announcements = $stmt->fetchAll();

        echo json_encode(['success' => true, 'announcements' => $announcements]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function createNotification($userId, $userType, $title, $message, $type = 'info')
{
    global $pdo;

    try {
        $stmt = $pdo->prepare("INSERT INTO notifications (user_id, user_type, title, message, type) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $userType, $title, $message, $type]);
    } catch (Exception $e) {
        error_log('Notification error: ' . $e->getMessage());
    }
}

function truncateText($text, $length = 100)
{
    if (strlen($text) <= $length)
        return $text;
    return substr($text, 0, $length) . '...';
}

function notifyStudentsBySubject($subject, $title, $message, $type = 'info')
{
    global $pdo;
    try {
        // Find all students who have this subject in their required_subjects (JSON array)
        $stmt = $pdo->query("SELECT id, required_subjects FROM students");
        $students = $stmt->fetchAll();

        foreach ($students as $student) {
            $subjects = json_decode($student['required_subjects'], true) ?: [];
            if (in_array($subject, $subjects)) {
                createNotification($student['id'], 'student', $title, $message, $type);
            }
        }
    } catch (Exception $e) {
        error_log('Notify students error: ' . $e->getMessage());
    }
}

function notifyAllStudents($title, $message, $type = 'info')
{
    global $pdo;
    try {
        $stmt = $pdo->query("SELECT id FROM students");
        $ids = $stmt->fetchAll(PDO::FETCH_COLUMN);
        foreach ($ids as $id) {
            createNotification($id, 'student', $title, $message, $type);
        }
    } catch (Exception $e) {
        error_log('Notify all students error: ' . $e->getMessage());
    }
}

// ========== REGISTRATION FUNCTIONS ==========

function registerStudent()
{
    global $pdo;

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $age = isset($_POST['age']) ? (int) $_POST['age'] : 0;
    $parentGuardian = isset($_POST['parent_guardian_name']) ? trim($_POST['parent_guardian_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';
    $gradeLevel = isset($_POST['grade_level']) ? $_POST['grade_level'] : '';
    $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : [];

    if (empty($fullName) || $age < 5 || $age > 25 || empty($parentGuardian) || empty($phone) || empty($educationLevel) || empty($gradeLevel) || empty($requiredSubjects)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan dengan betul']);
        return;
    }

    $username = generateUsername($fullName);
    $password = password_hash('pelajar', PASSWORD_DEFAULT);

    try {
        $stmt = $pdo->prepare("INSERT INTO students (username, password, full_name, age, parent_guardian_name, phone, education_level, grade_level, required_subjects) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$username, $password, $fullName, $age, $parentGuardian, $phone, $educationLevel, $gradeLevel, json_encode($requiredSubjects)]);

        echo json_encode(['success' => true, 'message' => 'Pendaftaran berjaya! Sila log masuk.', 'username' => $username, 'password' => 'pelajar']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function registerTeacher()
{
    global $pdo;

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';
    $gradeLevel = isset($_POST['grade_level']) ? $_POST['grade_level'] : '';
    $subjects = isset($_POST['subjects']) ? $_POST['subjects'] : [];

    if (empty($fullName) || empty($phone) || empty($educationLevel) || empty($gradeLevel) || empty($subjects)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan']);
        return;
    }

    $username = generateUsername($fullName);
    $password = password_hash('tutor123', PASSWORD_DEFAULT);

    try {
        $stmt = $pdo->prepare("INSERT INTO teachers (username, password, full_name, phone, education_level, grade_level, subjects) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$username, $password, $fullName, $phone, $educationLevel, $gradeLevel, json_encode($subjects)]);

        echo json_encode(['success' => true, 'message' => 'Pendaftaran berjaya! Sila log masuk.', 'username' => $username, 'password' => 'tutor123']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function checkUsername()
{
    global $pdo;

    $username = isset($_GET['username']) ? trim($_GET['username']) : '';

    if (empty($username)) {
        echo json_encode(['success' => false, 'message' => 'Nama pengguna diperlukan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM admin WHERE username = ? UNION ALL SELECT COUNT(*) FROM teachers WHERE username = ? UNION ALL SELECT COUNT(*) FROM students WHERE username = ?");
        $stmt->execute([$username, $username, $username]);
        $counts = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $total = array_sum($counts);

        if ($total > 0) {
            echo json_encode(['success' => true, 'available' => false, 'message' => 'Nama pengguna sudah digunakan']);
        } else {
            echo json_encode(['success' => true, 'available' => true, 'message' => 'Nama pengguna tersedia']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== HELPER FUNCTIONS ==========

function generateUsername($name)
{
    global $pdo;

    // Ambil nama pertama sahaja dan bersihkan
    $parts = explode(' ', trim($name));
    $firstName = preg_replace('/[^a-zA-Z]/', '', strtolower($parts[0]));

    // Jika nama terlalu pendek, guna gabungan
    if (strlen($firstName) < 3 && isset($parts[1])) {
        $firstName .= preg_replace('/[^a-zA-Z]/', '', strtolower($parts[1]));
    }

    $baseName = substr($firstName, 0, 10);

    // 1. Cuba nama asal dulu
    $stmt = $pdo->prepare("SELECT id FROM students WHERE username = ? UNION SELECT id FROM teachers WHERE username = ? UNION SELECT id FROM admin WHERE username = ?");
    $stmt->execute([$baseName, $baseName, $baseName]);
    if (!$stmt->fetch()) {
        return $baseName;
    }

    // 2. Jika sudah wujud, cuba tambah 1, 2, 3...
    $counter = 1;
    while ($counter <= 50) {
        $username = $baseName . $counter;
        $stmt = $pdo->prepare("SELECT id FROM students WHERE username = ? UNION SELECT id FROM teachers WHERE username = ? UNION SELECT id FROM admin WHERE username = ?");
        $stmt->execute([$username, $username, $username]);
        if (!$stmt->fetch()) {
            return $username;
        }
        $counter++;
    }

    // 3. Fallback jika semua gagal
    return $baseName . rand(100, 999);
}

// ========== MODULE FUNCTIONS ==========

function getTeacherModules()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->prepare("SELECT * FROM modules WHERE teacher_id = ? ORDER BY id DESC");
        $stmt->execute([$_SESSION['user_id']]);
        $modules = $stmt->fetchAll();

        echo json_encode(['success' => true, 'modules' => $modules]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addModule()
{
    global $pdo;
    requireTeacher();

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $links = isset($_POST['links']) ? $_POST['links'] : [];
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $forAllStudents = isset($_POST['for_all_students']) ? $_POST['for_all_students'] : 'No';

    if (empty($title) || empty($subject)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi tajuk dan subjek']);
        return;
    }

    // Handle File Uploads
    if (isset($_FILES['module_attachments'])) {
        $uploadDir = __DIR__ . '/assets/uploads/modules/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }

        $files = $_FILES['module_attachments'];
        // Normalise files array structure
        $fileCount = is_array($files['name']) ? count($files['name']) : 0;

        if ($fileCount > 0) {
            for ($i = 0; $i < $fileCount; $i++) {
                if ($files['error'][$i] === UPLOAD_ERR_OK) {
                    $fileName = basename($files['name'][$i]);
                    $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                    $newFileName = time() . '_' . uniqid() . '.' . $fileExt;
                    $targetPath = $uploadDir . $newFileName;

                    if (move_uploaded_file($files['tmp_name'][$i], $targetPath)) {
                        $links[] = 'assets/uploads/modules/' . $newFileName;
                    }
                }
            }
        }
    }

    try {
        $teacherId = $_SESSION['user_id'];
        $linksJson = json_encode($links);
        $stmt = $pdo->prepare("INSERT INTO modules (title, description, content, links, subject, teacher_id, for_all_students) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$title, $description, $content, $linksJson, $subject, $teacherId, $forAllStudents]);

        $moduleId = $pdo->lastInsertId();

        // Handle specific students
        if ($forAllStudents === 'No' && isset($_POST['module_students']) && is_array($_POST['module_students'])) {
            $stmt = $pdo->prepare("INSERT INTO module_students (module_id, student_id) VALUES (?, ?)");
            foreach ($_POST['module_students'] as $sId) {
                $stmt->execute([$moduleId, $sId]);
                createNotification($sId, 'student', 'Modul Baru: ' . $title, 'Cikgu telah menambah modul baru khas untuk anda!', 'module');
            }
        } else {
            // Notify all students of this subject
            notifyStudentsBySubject($subject, 'Modul Baru: ' . $title, 'Cikgu telah memuat naik modul baru untuk subjek ' . $subject, 'module');
        }

        echo json_encode(['success' => true, 'message' => 'Modul berjaya ditambah', 'module_id' => $moduleId]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateModule()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $links = isset($_POST['links']) ? $_POST['links'] : [];
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $forAllStudents = isset($_POST['for_all_students']) ? $_POST['for_all_students'] : 'No';

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID modul tidak valid']);
        return;
    }

    // Handle File Uploads
    if (isset($_FILES['module_attachments'])) {
        $uploadDir = __DIR__ . '/assets/uploads/modules/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }

        $files = $_FILES['module_attachments'];
        $fileCount = is_array($files['name']) ? count($files['name']) : 0;

        if ($fileCount > 0) {
            for ($i = 0; $i < $fileCount; $i++) {
                if ($files['error'][$i] === UPLOAD_ERR_OK) {
                    $fileName = basename($files['name'][$i]);
                    $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                    $newFileName = time() . '_' . uniqid() . '.' . $fileExt;
                    $targetPath = $uploadDir . $newFileName;

                    if (move_uploaded_file($files['tmp_name'][$i], $targetPath)) {
                        $links[] = 'assets/uploads/modules/' . $newFileName;
                    }
                }
            }
        }
    }

    try {
        $teacherId = $_SESSION['user_id'];
        $linksJson = json_encode($links);
        $stmt = $pdo->prepare("UPDATE modules SET title = ?, description = ?, content = ?, links = ?, subject = ?, for_all_students = ? WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$title, $description, $content, $linksJson, $subject, $forAllStudents, $id, $teacherId]);

        // Update students if not for all
        if ($forAllStudents === 'No') {
            // Delete existing first
            $stmt = $pdo->prepare("DELETE FROM module_students WHERE module_id = ?");
            $stmt->execute([$id]);

            if (isset($_POST['module_students']) && is_array($_POST['module_students'])) {
                $stmt = $pdo->prepare("INSERT INTO module_students (module_id, student_id) VALUES (?, ?)");
                foreach ($_POST['module_students'] as $sId) {
                    $stmt->execute([$id, $sId]);
                }
            }
        } else {
            // If changed to Yes, clear any specific assignments
            $stmt = $pdo->prepare("DELETE FROM module_students WHERE module_id = ?");
            $stmt->execute([$id]);
        }

        echo json_encode(['success' => true, 'message' => 'Modul berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteModule()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID modul tidak valid']);
        return;
    }

    try {
        $teacherId = $_SESSION['user_id'];
        // Cascade delete if your DB supports it, otherwise manual here
        $stmt = $pdo->prepare("DELETE FROM module_students WHERE module_id = ?");
        $stmt->execute([$id]);

        $stmt = $pdo->prepare("DELETE FROM modules WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$id, $teacherId]);

        echo json_encode(['success' => true, 'message' => 'Modul berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentModules()
{
    global $pdo;
    requireStudent();

    try {
        $studentId = $_SESSION['user_id'];

        // Modules that are for all students OR specifically for this student
        $stmt = $pdo->prepare("
            SELECT DISTINCT m.*, t.full_name as teacher_name 
            FROM modules m 
            JOIN teachers t ON m.teacher_id = t.id 
            LEFT JOIN module_students ms ON m.id = ms.module_id
            WHERE m.status = 'Active' 
            AND (m.for_all_students = 'Yes' OR ms.student_id = ?)
            ORDER BY m.id DESC
        ");
        $stmt->execute([$studentId]);
        $modules = $stmt->fetchAll();

        echo json_encode(['success' => true, 'modules' => $modules]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function enrollStudentModule()
{
    global $pdo;
    requireTeacher();

    $moduleId = isset($_POST['module_id']) ? (int) $_POST['module_id'] : 0;
    $studentIds = isset($_POST['student_ids']) ? $_POST['student_ids'] : [];

    if ($moduleId === 0 || empty($studentIds)) {
        echo json_encode(['success' => false, 'message' => 'Sila pilih modul dan pelajar']);
        return;
    }

    try {
        $stmt = $pdo->prepare("INSERT IGNORE INTO module_students (module_id, student_id) VALUES (?, ?)");
        foreach ($studentIds as $studentId) {
            $stmt->execute([$moduleId, $studentId]);
        }

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya didaftarkan ke modul']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== SCHEDULE FUNCTIONS ==========

function getTeacherSchedule()
{
    global $pdo;
    requireTeacher();

    $view = isset($_GET['view']) ? $_GET['view'] : 'own'; // 'own' or 'all'

    try {
        if ($view === 'all') {
            // Get ALL active schedules from ALL teachers
            $sql = "SELECT s.*, t.full_name as teacher_name 
                    FROM schedule s 
                    LEFT JOIN teachers t ON s.teacher_id = t.id 
                    WHERE s.status = 'Active' 
                    ORDER BY s.schedule_date, s.start_time";
            $stmt = $pdo->prepare($sql);
            $stmt->execute();
        } else {
            // Get ONLY logged-in teacher's schedules
            $sql = "SELECT s.*, t.full_name as teacher_name 
                    FROM schedule s 
                    LEFT JOIN teachers t ON s.teacher_id = t.id 
                    WHERE s.teacher_id = ? AND s.status = 'Active' 
                    ORDER BY s.schedule_date, s.start_time";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$_SESSION['user_id']]);
        }

        $schedules = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode(['success' => true, 'schedules' => $schedules]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addSchedule()
{
    global $pdo;
    requireTeacher();

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $dayOfWeek = isset($_POST['day_of_week']) ? $_POST['day_of_week'] : '';
    $scheduleDate = isset($_POST['schedule_date']) ? $_POST['schedule_date'] : '';
    $startTime = isset($_POST['start_time']) ? $_POST['start_time'] : '';
    $duration = isset($_POST['duration_minutes']) ? (int) $_POST['duration_minutes'] : 60;
    $isRecurring = isset($_POST['is_recurring']) ? $_POST['is_recurring'] : 'No';
    $selectedStudents = isset($_POST['selected_students']) ? $_POST['selected_students'] : null;

    if (empty($title) || empty($subject) || empty($startTime)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan']);
        return;
    }

    try {
        $endTime = date('H:i:s', strtotime($startTime . ' +' . $duration . ' minutes'));

        $stmt = $pdo->prepare("INSERT INTO schedule (teacher_id, day_of_week, schedule_date, start_time, end_time, subject, description, title, selected_students) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$_SESSION['user_id'], $dayOfWeek, $scheduleDate, $startTime, $endTime, $subject, $description, $title, $selectedStudents]);

        // Notify students
        notifyStudentsBySubject($subject, 'Jadual Baru: ' . $subject, 'Slot baru telah ditambah pada ' . ($scheduleDate ?: $dayOfWeek), 'schedule');

        echo json_encode(['success' => true, 'message' => 'Jadual berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateSchedule()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $dayOfWeek = isset($_POST['day_of_week']) ? $_POST['day_of_week'] : '';
    $scheduleDate = isset($_POST['schedule_date']) ? $_POST['schedule_date'] : '';
    $startTime = isset($_POST['start_time']) ? $_POST['start_time'] : '';
    $duration = isset($_POST['duration_minutes']) ? (int) $_POST['duration_minutes'] : 60;
    $isRecurring = isset($_POST['is_recurring']) ? $_POST['is_recurring'] : 'No';
    $selectedStudents = isset($_POST['selected_students']) ? $_POST['selected_students'] : null;
    $joinLink = isset($_POST['join_link']) ? trim($_POST['join_link']) : ''; // Also support join_link update here if full update

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID jadual tidak valid']);
        return;
    }

    try {
        // If join_link is provided, update it too, otherwise keep existing logic for other fields
        // Since updateClass calls this via alias (wait, no updateClass calls updateSchedule? No, updateClass does its own thing for link/desc)
        // But if user updates from SCHEDULE tab, we should also save selected_students.

        $sql = "UPDATE schedule SET title = ?, description = ?, subject = ?, day_of_week = ?, schedule_date = ?, start_time = ?, duration_minutes = ?, is_recurring = ?, selected_students = ?";
        $params = [$title, $description, $subject, $dayOfWeek, $scheduleDate, $startTime, $duration, $isRecurring, $selectedStudents];

        if (!empty($joinLink)) {
            $sql .= ", join_link = ?";
            $params[] = $joinLink;
        }

        $sql .= " WHERE id = ? AND teacher_id = ?";
        $params[] = $id;
        $params[] = $_SESSION['user_id'];

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        echo json_encode(['success' => true, 'message' => 'Jadual masa berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteSchedule()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID jadual tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE schedule SET status = 'Inactive' WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$id, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Jadual masa berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentSchedule()
{
    global $pdo;
    requireStudent();

    $studentId = $_SESSION['user_id'];

    try {
        // 1. Get Student Info (Mentor & Required Subjects)
        $stmt = $pdo->prepare("SELECT enrolled_by, required_subjects FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $studentInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$studentInfo) {
            echo json_encode(['success' => true, 'schedules' => []]);
            return;
        }

        $mentorId = $studentInfo['enrolled_by'];
        $requiredSubjects = json_decode($studentInfo['required_subjects'] ?: '[]', true);
        if (!is_array($requiredSubjects)) $requiredSubjects = [];

        // 2. Get Enrolled Subjects (Teacher-Specific)
        $stmt = $pdo->prepare("SELECT teacher_id, subject FROM student_enrollments WHERE student_id = ? AND status = 'Approved'");
        $stmt->execute([$studentId]);
        $enrollments = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Build a map of Teacher ID -> Allowed Subjects
        $teacherSubjectsMap = [];
        
        // Add Mentor (Allowed subjects = Student's Required Subjects)
        if ($mentorId) {
            $teacherSubjectsMap[$mentorId] = $requiredSubjects;
        }

        // Add Enrollments (Specific Teacher-Subject pairs)
        foreach ($enrollments as $enroll) {
            if (!isset($teacherSubjectsMap[$enroll['teacher_id']])) {
                $teacherSubjectsMap[$enroll['teacher_id']] = [];
            }
            // Add subject if not already there
            if (!in_array($enroll['subject'], $teacherSubjectsMap[$enroll['teacher_id']])) {
                $teacherSubjectsMap[$enroll['teacher_id']][] = $enroll['subject'];
            }
        }

        // 3. Fetch All Schedules from these teachers
        if (empty($teacherSubjectsMap)) {
            echo json_encode(['success' => true, 'schedules' => []]);
            return;
        }

        $teacherIds = array_keys($teacherSubjectsMap);
        $placeholders = implode(',', array_fill(0, count($teacherIds), '?'));
        
        $sql = "
            SELECT s.*, t.full_name as teacher_name,
            CASE WHEN t.id = ? THEN 1 ELSE 0 END as is_mentor
            FROM schedule s
            JOIN teachers t ON s.teacher_id = t.id
            WHERE s.status = 'Active' 
            AND s.teacher_id IN ($placeholders)
            ORDER BY s.schedule_date, s.start_time
        ";

        $params = array_merge([$mentorId], $teacherIds);
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $allSchedules = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 4. Filter Schedules
        $filteredSchedules = [];
        foreach ($allSchedules as $sch) {
            $tid = $sch['teacher_id'];
            $subj = $sch['subject'];
            $selectedStudents = $sch['selected_students'];

            $isRelevant = false;

            // Check specific selection first
            if (!empty($selectedStudents)) {
                $selectedIds = json_decode($selectedStudents, true);
                // Handle case where it might be a string "null" or invalid JSON
                if (is_array($selectedIds) && in_array((string)$studentId, array_map('strval', $selectedIds))) {
                    $isRelevant = true;
                }
            } 
            
            // If not specifically selected, check if open class AND subject matches
            if (!$isRelevant && empty($selectedStudents)) {
                if (isset($teacherSubjectsMap[$tid]) && in_array($subj, $teacherSubjectsMap[$tid])) {
                    $isRelevant = true;
                }
            }

            if ($isRelevant) {
                $filteredSchedules[] = $sch;
            }
        }

        echo json_encode(['success' => true, 'schedules' => $filteredSchedules]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== ONLINE CLASS FUNCTIONS ==========

// Unified Schedule & Class System

function getTeacherClasses()
{
    global $pdo;
    requireTeacher();

    // Fetch from schedule table instead of online_classes to ensure sync
    try {
        $stmt = $pdo->prepare("SELECT * FROM schedule WHERE teacher_id = ? AND status = 'Active' ORDER BY schedule_date DESC, start_time ASC");
        $stmt->execute([$_SESSION['user_id']]);
        $classes = $stmt->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode(['success' => true, 'classes' => $classes]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// Alias addClass to addSchedule for consistency, or keep separate if strictly for online
function addClass()
{
    addSchedule();
}

// Update Class (Focus on Link & Desc)
function updateClass()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $joinLink = isset($_POST['join_link']) ? trim($_POST['join_link']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    // Fix: Capture selected_students if provided (JSON string)
    $selectedStudents = isset($_POST['selected_students']) ? $_POST['selected_students'] : null;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID kelas tidak valid']);
        return;
    }

    try {
        // Update link, description AND selected_students if provided
        $sql = "UPDATE schedule SET join_link = ?, description = ?";
        $params = [$joinLink, $description];

        if ($selectedStudents !== null) {
            $sql .= ", selected_students = ?";
            $params[] = $selectedStudents;
        }

        $sql .= " WHERE id = ? AND teacher_id = ?";
        $params[] = $id;
        $params[] = $_SESSION['user_id'];

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        echo json_encode(['success' => true, 'message' => 'Kelas berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}


function deleteClass()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID kelas tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE online_classes SET status = 'Cancelled' WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$id, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Kelas online berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentClasses()
{
    global $pdo;
    requireStudent();

    $studentId = $_SESSION['user_id'];

    try {
        // 1. Get Student Info (Mentor & Required Subjects)
        $stmt = $pdo->prepare("SELECT enrolled_by, required_subjects FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $studentInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$studentInfo) {
            echo json_encode(['success' => true, 'classes' => []]);
            return;
        }

        $mentorId = $studentInfo['enrolled_by'];
        $requiredSubjects = json_decode($studentInfo['required_subjects'] ?: '[]', true);
        if (!is_array($requiredSubjects)) {
            $requiredSubjects = [];
        }

        // 2. Get Enrolled Subjects (Teacher-Specific)
        $stmt = $pdo->prepare("SELECT teacher_id, subject FROM student_enrollments WHERE student_id = ? AND status = 'Approved'");
        $stmt->execute([$studentId]);
        $enrollments = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Build a map of Teacher ID -> Allowed Subjects
        $teacherSubjectsMap = [];
        
        // Add Mentor (Allowed subjects = Student's Required Subjects)
        if ($mentorId) {
            $teacherSubjectsMap[$mentorId] = $requiredSubjects;
        }

        // Add Enrollments (Specific Teacher-Subject pairs)
        foreach ($enrollments as $enroll) {
            if (!isset($teacherSubjectsMap[$enroll['teacher_id']])) {
                $teacherSubjectsMap[$enroll['teacher_id']] = [];
            }
            // Add subject if not already there
            if (!in_array($enroll['subject'], $teacherSubjectsMap[$enroll['teacher_id']])) {
                $teacherSubjectsMap[$enroll['teacher_id']][] = $enroll['subject'];
            }
        }

        // 3. Fetch All Schedules from these teachers
        if (empty($teacherSubjectsMap)) {
            echo json_encode(['success' => true, 'classes' => []]);
            return;
        }

        $teacherIds = array_keys($teacherSubjectsMap);
        $placeholders = implode(',', array_fill(0, count($teacherIds), '?'));
        
        // Fetch from SCHEDULE table instead of online_classes
        // Map columns to match what frontend expects (class_date, title, etc.)
        $sql = "
            SELECT 
                s.id,
                s.schedule_date as class_date, 
                s.start_time, 
                s.end_time, 
                s.subject, 
                s.description, 
                s.join_link,
                s.selected_students,
                s.status,
                s.teacher_id,
                t.full_name as teacher_name,
                s.subject as title
            FROM schedule s
            JOIN teachers t ON s.teacher_id = t.id
            WHERE s.status = 'Active' 
            AND s.teacher_id IN ($placeholders)
            ORDER BY s.schedule_date DESC, s.start_time DESC
        ";

        $params = $teacherIds; 
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $allSchedules = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 4. Filter Schedules
        $filteredClasses = [];
        foreach ($allSchedules as $sch) {
            $tid = $sch['teacher_id'];
            $subj = $sch['subject'];
            $selectedStudents = $sch['selected_students'];

            $isRelevant = false;

            // Check specific selection first
            if (!empty($selectedStudents)) {
                $selectedIds = json_decode($selectedStudents, true);
                if (is_array($selectedIds) && in_array((string)$studentId, array_map('strval', $selectedIds))) {
                    $isRelevant = true;
                }
            } 
            
            // If not specifically selected, check if open class AND subject matches
            if (!$isRelevant && empty($selectedStudents)) {
                if (isset($teacherSubjectsMap[$tid]) && in_array($subj, $teacherSubjectsMap[$tid])) {
                    $isRelevant = true;
                }
            }

            if ($isRelevant) {
                $filteredClasses[] = $sch;
            }
        }

        echo json_encode(['success' => true, 'classes' => $filteredClasses]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function enrollStudentClass()
{
    global $pdo;
    requireTeacher();

    $classId = isset($_POST['class_id']) ? (int) $_POST['class_id'] : 0;
    $studentIds = isset($_POST['student_ids']) ? $_POST['student_ids'] : [];

    if ($classId === 0 || empty($studentIds)) {
        echo json_encode(['success' => false, 'message' => 'Sila pilih kelas dan pelajar']);
        return;
    }

    try {
        $stmt = $pdo->prepare("INSERT IGNORE INTO class_students (class_id, student_id) VALUES (?, ?)");
        foreach ($studentIds as $studentId) {
            $stmt->execute([$classId, $studentId]);
        }

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya didaftarkan ke kelas']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function classComment()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || !isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    $classId = isset($_POST['class_id']) ? (int) $_POST['class_id'] : 0;
    $comment = isset($_POST['comment']) ? trim($_POST['comment']) : '';
    $type = isset($_POST['type']) ? $_POST['type'] : '';

    if ($classId === 0 || empty($comment)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi komen']);
        return;
    }

    try {
        if ($_SESSION['user_type'] === 'teacher') {
            $stmt = $pdo->prepare("UPDATE class_students SET teacher_notes = ? WHERE class_id = ? AND student_id IN (SELECT id FROM students WHERE enrolled_by = ?)");
            $stmt->execute([$comment, $classId, $_SESSION['user_id']]);
        } else {
            $stmt = $pdo->prepare("UPDATE class_students SET student_comment = ? WHERE class_id = ? AND student_id = ?");
            $stmt->execute([$comment, $classId, $_SESSION['user_id']]);
        }

        echo json_encode(['success' => true, 'message' => 'Komen berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== FORUM FUNCTIONS ==========

function getTeacherForums()
{
    global $pdo;
    requireTeacher();

    $teacherId = $_SESSION['user_id'];

    try {
        // Teachers see: 
        // 1. Forums they created
        // 2. Forums shared with 'all'
        // 3. Forums shared with 'teachers'
        // 4. Forums where their ID is in the specific list
        $stmt = $pdo->prepare("
            SELECT f.*, t.full_name as teacher_name, t.profile_emoji as teacher_emoji
            FROM forums f
            JOIN teachers t ON f.teacher_id = t.id
            WHERE f.status = 'Active' 
            AND (f.teacher_id = ? OR f.target_audience = 'all' OR f.target_audience = 'teachers')
            ORDER BY f.is_pinned DESC, f.created_at DESC
        ");
        $stmt->execute([$teacherId]);
        $allForums = $stmt->fetchAll();

        // Filter for specific teachers
        $filtered = [];
        foreach ($allForums as $f) {
            if ($f['teacher_id'] == $teacherId || $f['target_audience'] == 'all' || $f['target_audience'] == 'teachers') {
                $filtered[] = $f;
            } else {
                try {
                    $ids = json_decode($f['target_audience'], true);
                    if (is_array($ids) && in_array((string) $teacherId, $ids)) {
                        $filtered[] = $f;
                    }
                } catch (Exception $e) {
                }
            }
        }

        echo json_encode(['success' => true, 'forums' => $filtered]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addForum()
{
    global $pdo;
    requireTeacher();

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $link = isset($_POST['link']) ? trim($_POST['link']) : '';
    $targetAudience = isset($_POST['target_audience']) ? $_POST['target_audience'] : 'all';
    $isPinned = isset($_POST['is_pinned']) ? $_POST['is_pinned'] : 'No';

    // Determine which IDs to use based on target audience type
    $idsRaw = '[]';
    if ($targetAudience === 'specific' || $targetAudience === 'specific_students') {
        // If JS sends it as selected_students (JSON string) or forum_students (array)
        if (isset($_POST['selected_students'])) {
            $idsRaw = $_POST['selected_students']; // Assuming it's already a JSON string from the UI
        } else if (isset($_POST['forum_students'])) {
            $idsRaw = json_encode($_POST['forum_students']);
        }
    }

    if (empty($title) || empty($content) || empty($subject)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan']);
        return;
    }

    // Handle File Upload
    $attachmentPath = null;
    $attachmentType = null;

    if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['attachment'];
        $fileName = basename($file['name']);
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

        $allowed = ['jpg', 'jpeg', 'png', 'pdf'];

        if (!in_array($fileExt, $allowed)) {
            echo json_encode(['success' => false, 'message' => 'Hanya fail JPG, PNG dan PDF dibenarkan']);
            return;
        }

        $uploadDir = __DIR__ . '/assets/uploads/forum/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }

        $newFileName = time() . '_' . uniqid() . '.' . $fileExt;
        $targetPath = $uploadDir . $newFileName;

        if (move_uploaded_file($file['tmp_name'], $targetPath)) {
            $attachmentPath = 'assets/uploads/forum/' . $newFileName;
            $attachmentType = $fileExt;
        }
    }

    try {
        $teacherId = $_SESSION['user_id'];
        $finalAudience = ($targetAudience === 'all' || $targetAudience === 'teachers' || $targetAudience === 'students') ? $targetAudience : $idsRaw;

        $stmt = $pdo->prepare("INSERT INTO forums (teacher_id, subject, title, content, link, is_pinned, target_audience, attachment, attachment_type) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$teacherId, $subject, $title, $content, $link, $isPinned, $finalAudience, $attachmentPath, $attachmentType]);

        // Notify students
        if ($finalAudience === 'all') {
            notifyStudentsBySubject($subject, 'Forum Baru: ' . $title, 'Jom berbincang tentang ' . $subject . ' di forum!', 'forum');
        } else {
            $ids = json_decode($finalAudience, true);
            if (is_array($ids)) {
                foreach ($ids as $sId) {
                    createNotification($sId, 'student', 'Forum Baru: ' . $title, 'Guru telah menambah topik forum untuk anda!', 'forum');
                }
            }
        }

        echo json_encode(['success' => true, 'message' => 'Forum berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateForum()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $subject = isset($_POST['subject']) ? $_POST['subject'] : '';
    $isPinned = isset($_POST['is_pinned']) ? $_POST['is_pinned'] : 'No';
    $link = isset($_POST['link']) ? trim($_POST['link']) : '';
    $targetAudience = isset($_POST['target_audience']) ? $_POST['target_audience'] : 'all';

    // Determine which IDs to use based on target audience type
    $idsRaw = '[]';
    if ($targetAudience === 'specific' || $targetAudience === 'specific_students') {
        if (isset($_POST['selected_students'])) {
            $idsRaw = $_POST['selected_students'];
        } else if (isset($_POST['forum_students'])) {
            $idsRaw = json_encode($_POST['forum_students']);
        }
    }

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID forum tidak valid']);
        return;
    }

    try {
        $finalAudience = ($targetAudience === 'all' || $targetAudience === 'teachers' || $targetAudience === 'students') ? $targetAudience : $idsRaw;

        // Handle File Upload for Update
        $attachmentSql = "";
        $params = [$title, $content, $subject, $isPinned, $link, $finalAudience];

        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['attachment'];
            $fileName = basename($file['name']);
            $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $allowed = ['jpg', 'jpeg', 'png', 'pdf'];

            if (in_array($fileExt, $allowed)) {
                $uploadDir = __DIR__ . '/assets/uploads/forum/';
                if (!file_exists($uploadDir))
                    mkdir($uploadDir, 0777, true);

                $newFileName = time() . '_' . uniqid() . '.' . $fileExt;
                if (move_uploaded_file($file['tmp_name'], $uploadDir . $newFileName)) {
                    $attachmentSql = ", attachment = ?, attachment_type = ?";
                    $params[] = 'assets/uploads/forum/' . $newFileName;
                    $params[] = $fileExt;
                }
            }
        }

        $params[] = $id;
        $params[] = $_SESSION['user_id'];

        $stmt = $pdo->prepare("UPDATE forums SET title = ?, content = ?, subject = ?, is_pinned = ?, link = ?, target_audience = ?" . $attachmentSql . " WHERE id = ? AND teacher_id = ?");
        $stmt->execute($params);

        echo json_encode(['success' => true, 'message' => 'Forum berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteForum()
{
    global $pdo;
    requireTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID forum tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE forums SET status = 'Inactive' WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$id, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Forum berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentForums()
{
    global $pdo;
    requireStudent();

    $studentId = $_SESSION['user_id'];

    try {
        // Fetch all active forums and filter in PHP for JSON target_audience
        $stmt = $pdo->prepare("
            SELECT DISTINCT f.*, t.full_name as teacher_name, t.profile_emoji as teacher_emoji
            FROM forums f
            JOIN teachers t ON f.teacher_id = t.id
            WHERE f.status = 'Active' 
            AND (
                t.id = (SELECT enrolled_by FROM students WHERE id = ?)
                OR
                t.id IN (SELECT teacher_id FROM student_enrollments WHERE student_id = ? AND status = 'Approved')
            )
            ORDER BY f.is_pinned DESC, f.created_at DESC
        ");
        $stmt->execute([$studentId, $studentId]);
        $allForums = $stmt->fetchAll();

        $filteredForums = [];
        foreach ($allForums as $forum) {
            $audience = $forum['target_audience'];
            if ($audience === 'all' || $audience === 'students') {
                $filteredForums[] = $forum;
            } else {
                try {
                    $ids = json_decode($audience, true);
                    if (is_array($ids) && in_array((string) $studentId, $ids)) {
                        $filteredForums[] = $forum;
                    }
                } catch (Exception $e) {
                    // Fallback or ignore malformed JSON
                }
            }
        }

        echo json_encode(['success' => true, 'forums' => $filteredForums]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addForumReply()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || !isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    $forumId = isset($_POST['forum_id']) ? (int) $_POST['forum_id'] : 0;
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';

    if ($forumId === 0 || empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi kandungan reply']);
        return;
    }

    // Handle File Upload
    $attachmentPath = null;
    $attachmentType = null;

    if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['attachment'];
        $fileName = basename($file['name']);
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

        $allowed = ['jpg', 'jpeg', 'png', 'pdf'];

        if (!in_array($fileExt, $allowed)) {
            echo json_encode(['success' => false, 'message' => 'Hanya fail JPG, PNG dan PDF dibenarkan']);
            return;
        }

        // Create directory if not exists
        $uploadDir = __DIR__ . '/assets/uploads/forum/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }

        $newFileName = time() . '_' . uniqid() . '.' . $fileExt;
        $targetPath = $uploadDir . $newFileName;

        if (move_uploaded_file($file['tmp_name'], $targetPath)) {
            $attachmentPath = 'assets/uploads/forum/' . $newFileName;
            $attachmentType = $fileExt;
        } else {
            echo json_encode(['success' => false, 'message' => 'Gagal memuat naik fail']);
            return;
        }
    }

    try {
        $stmt = $pdo->prepare("INSERT INTO forum_replies (forum_id, user_id, user_type, content, attachment, attachment_type) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$forumId, $_SESSION['user_id'], $_SESSION['user_type'], $content, $attachmentPath, $attachmentType]);

        echo json_encode(['success' => true, 'message' => 'Reply berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getForumReplies()
{
    global $pdo;

    $forumId = isset($_GET['forum_id']) ? (int) $_GET['forum_id'] : 0;

    if ($forumId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID forum tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("
            SELECT fr.*, 
                   CASE 
                     WHEN fr.user_type = 'teacher' THEN t.full_name 
                     ELSE s.full_name 
                   END as user_name,
                   CASE 
                     WHEN fr.user_type = 'teacher' THEN t.profile_emoji 
                     ELSE s.profile_emoji 
                   END as user_emoji
            FROM forum_replies fr
            LEFT JOIN teachers t ON fr.user_id = t.id AND fr.user_type = 'teacher'
            LEFT JOIN students s ON fr.user_id = s.id AND fr.user_type = 'student'
            WHERE fr.forum_id = ?
            ORDER BY fr.created_at ASC
        ");
        $stmt->execute([$forumId]);
        $replies = $stmt->fetchAll();

        echo json_encode(['success' => true, 'replies' => $replies]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== ASSIGNMENT/TUGASAN FUNCTIONS ==========

function addAssignment()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $teacherId = $_SESSION['user_id'];
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $moduleId = isset($_POST['module_id']) && $_POST['module_id'] !== '' ? (int) $_POST['module_id'] : null;
    $dueDate = isset($_POST['due_date']) && $_POST['due_date'] !== '' ? $_POST['due_date'] : null;
    $type = isset($_POST['type']) ? $_POST['type'] : 'Kerja';
    $emoji = isset($_POST['emoji']) ? $_POST['emoji'] : '📝';
    $links = isset($_POST['links']) ? $_POST['links'] : '[]';

    if (empty($title)) {
        echo json_encode(['success' => false, 'message' => 'Tajuk tugasan diperlukan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("INSERT INTO assignments (teacher_id, module_id, title, description, links, due_date, type, emoji) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$teacherId, $moduleId, $title, $description, $links, $dueDate, $type, $emoji]);

        echo json_encode(['success' => true, 'message' => 'Tugasan berjaya ditambah', 'id' => $pdo->lastInsertId()]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateAssignment()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $teacherId = $_SESSION['user_id'];
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $moduleId = isset($_POST['module_id']) && $_POST['module_id'] !== '' ? (int) $_POST['module_id'] : null;
    $dueDate = isset($_POST['due_date']) && $_POST['due_date'] !== '' ? $_POST['due_date'] : null;
    $type = isset($_POST['type']) ? $_POST['type'] : 'Kerja';
    $emoji = isset($_POST['emoji']) ? $_POST['emoji'] : '📝';
    $links = isset($_POST['links']) ? $_POST['links'] : '[]';
    $status = isset($_POST['status']) ? $_POST['status'] : 'Active';

    if ($id === 0 || empty($title)) {
        echo json_encode(['success' => false, 'message' => 'Data tidak lengkap']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE assignments SET module_id = ?, title = ?, description = ?, links = ?, due_date = ?, type = ?, emoji = ?, status = ? WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$moduleId, $title, $description, $links, $dueDate, $type, $emoji, $status, $id, $teacherId]);

        echo json_encode(['success' => true, 'message' => 'Tugasan berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteAssignment()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $teacherId = $_SESSION['user_id'];

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tugasan tidak sah']);
        return;
    }

    try {
        $stmt = $pdo->prepare("DELETE FROM assignments WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$id, $teacherId]);

        echo json_encode(['success' => true, 'message' => 'Tugasan berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getTeacherAssignments()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $teacherId = $_SESSION['user_id'];
    $moduleId = isset($_GET['module_id']) ? (int) $_GET['module_id'] : null;

    try {
        $sql = "SELECT a.*, m.title as module_title, 
                (SELECT COUNT(*) FROM assignment_comments ac WHERE ac.assignment_id = a.id) as comment_count
                FROM assignments a
                LEFT JOIN modules m ON a.module_id = m.id
                WHERE a.teacher_id = ?";
        $params = [$teacherId];

        if ($moduleId) {
            $sql .= " AND a.module_id = ?";
            $params[] = $moduleId;
        }

        $sql .= " ORDER BY a.created_at DESC";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $assignments = $stmt->fetchAll();

        echo json_encode(['success' => true, 'assignments' => $assignments]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getStudentAssignments()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'student') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $studentId = $_SESSION['user_id'];

    try {
        // Get assignments from teachers the student is enrolled with
        $stmt = $pdo->prepare("
            SELECT a.*, t.full_name as teacher_name, t.profile_emoji as teacher_emoji, m.title as module_title,
                   (SELECT COUNT(*) FROM assignment_comments ac WHERE ac.assignment_id = a.id) as comment_count
            FROM assignments a
            INNER JOIN teachers t ON a.teacher_id = t.id
            LEFT JOIN modules m ON a.module_id = m.id
            INNER JOIN student_enrollments se ON se.teacher_id = t.id AND se.student_id = ? AND se.status = 'Approved'
            WHERE a.status = 'Active'
            ORDER BY a.due_date ASC, a.created_at DESC
        ");
        $stmt->execute([$studentId]);
        $assignments = $stmt->fetchAll();

        echo json_encode(['success' => true, 'assignments' => $assignments]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addAssignmentComment()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || !isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Sila log masuk']);
        return;
    }

    $assignmentId = isset($_POST['assignment_id']) ? (int) $_POST['assignment_id'] : 0;
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $emoji = isset($_POST['emoji']) ? $_POST['emoji'] : null;

    if ($assignmentId === 0 || empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi komen']);
        return;
    }

    try {
        $stmt = $pdo->prepare("INSERT INTO assignment_comments (assignment_id, user_id, user_type, content, emoji) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$assignmentId, $_SESSION['user_id'], $_SESSION['user_type'], $content, $emoji]);

        echo json_encode(['success' => true, 'message' => 'Komen berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getAssignmentComments()
{
    global $pdo;

    $assignmentId = isset($_GET['assignment_id']) ? (int) $_GET['assignment_id'] : 0;

    if ($assignmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tugasan tidak sah']);
        return;
    }

    try {
        $stmt = $pdo->prepare("
            SELECT ac.*, 
                   CASE 
                     WHEN ac.user_type = 'teacher' THEN t.full_name 
                     ELSE s.full_name 
                   END as user_name,
                   CASE 
                     WHEN ac.user_type = 'teacher' THEN t.profile_emoji 
                     ELSE s.profile_emoji 
                   END as user_emoji
            FROM assignment_comments ac
            LEFT JOIN teachers t ON ac.user_id = t.id AND ac.user_type = 'teacher'
            LEFT JOIN students s ON ac.user_id = s.id AND ac.user_type = 'student'
            WHERE ac.assignment_id = ?
            ORDER BY ac.created_at ASC
        ");
        $stmt->execute([$assignmentId]);
        $comments = $stmt->fetchAll();

        echo json_encode(['success' => true, 'comments' => $comments]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== AUTO-ASSIGNMENT FUNCTIONS ==========

/**
 * Register student with auto-assignment to teachers
 * Handles multi-subject students by assigning each subject to a separate teacher
 */
function registerStudentWithAssign()
{
    global $pdo;

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $age = isset($_POST['age']) ? (int) $_POST['age'] : 0;
    $parentGuardian = isset($_POST['parent_guardian_name']) ? trim($_POST['parent_guardian_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';
    $gradeLevel = isset($_POST['grade_level']) ? $_POST['grade_level'] : '';
    $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : [];
    $customSubject = isset($_POST['custom_subject']) ? trim($_POST['custom_subject']) : '';

    $enrolledBy = isset($_POST['enrolled_by']) ? (int) $_POST['enrolled_by'] : null;

    if (empty($fullName) || $age < 5 || $age > 25 || empty($parentGuardian) || empty($phone) || empty($educationLevel) || empty($gradeLevel) || empty($requiredSubjects)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan dengan betul']);
        return;
    }

    $username = generateUsername($fullName);
    $password = password_hash('pelajar', PASSWORD_DEFAULT);

    try {
        // Insert student
        $stmt = $pdo->prepare("INSERT INTO students (username, password, full_name, age, parent_guardian_name, phone, education_level, grade_level, required_subjects, enrolled_by, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Active')");
        $stmt->execute([$username, $password, $fullName, $age, $parentGuardian, $phone, $educationLevel, $gradeLevel, json_encode($requiredSubjects), $enrolledBy]);

        $studentId = $pdo->lastInsertId();

        // Auto-assign to teachers for each subject (handles multi-subject students)
        $enrollments = [];
        foreach ($requiredSubjects as $subject) {
            $assigned = autoAssignStudentToTeacher($studentId, $subject);
            $enrollments[] = $assigned;
        }

        // Handle custom subject
        if (!empty($customSubject)) {
            // Create a subject request for custom subject
            $stmt = $pdo->prepare("INSERT INTO subject_requests (student_id, requested_subject, status) VALUES (?, ?, 'Pending')");
            $stmt->execute([$studentId, $customSubject]);

            $enrollments[] = [
                'subject' => $customSubject,
                'status' => 'Menunggu Guru'
            ];
        }

        // Notify teachers of new enrollments
        foreach ($enrollments as $enrollment) {
            if ($enrollment['assigned'] ?? false) {
                createNotification($enrollment['teacher_id'], 'teacher', 'Pendaftaran Pelajar Baru', $fullName . ' telah mendaftar untuk subjek ' . $enrollment['subject'], 'enrollment');
            }
        }

        // Build response message
        $assignedCount = count(array_filter($enrollments, function ($e) {
            return $e['assigned'] ?? false;
        }));
        $message = "Pendaftaran berjaya! ";
        if ($assignedCount > 0) {
            $message .= $assignedCount . " subjek telah diassign kepada guru. ";
        }
        $message .= "Menunggu kelulusan guru.";

        echo json_encode([
            'success' => true,
            'message' => $message,
            'username' => $username,
            'password' => 'pelajar',
            'enrollments' => $enrollments
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}


/**
 * Sync all teacher students to ensure enrollments exist
 * Useful for legacy data or manual additions that missed enrollment entries
 */
function syncAllTeacherStudents()
{
    global $pdo;
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $teacherId = $_SESSION['user_id'];

    try {
        // 1. Find students where enrolled_by = teacherId but no enrollment record exists
        $stmt = $pdo->prepare("
            SELECT s.* 
            FROM students s
            LEFT JOIN student_enrollments se ON s.id = se.student_id AND se.teacher_id = ?
            WHERE s.enrolled_by = ? AND se.id IS NULL
        ");
        $stmt->execute([$teacherId, $teacherId]);
        $missingStudents = $stmt->fetchAll();

        $syncedCount = 0;
        foreach ($missingStudents as $student) {
            $required_subjects = $student['required_subjects'];
            $subjects = [];

            if (empty($required_subjects)) {
                $subjects = ['Am/Umum'];
            } else {
                try {
                    $subjects = json_decode($required_subjects, true);
                    if (!is_array($subjects))
                        $subjects = [$required_subjects];
                } catch (Exception $e) {
                    $subjects = [$required_subjects];
                }
            }

            if (empty($subjects))
                $subjects = ['Am/Umum'];

            foreach ($subjects as $subject) {
                // Double check if enrollment exists (just in case)
                $check = $pdo->prepare("SELECT id FROM student_enrollments WHERE student_id = ? AND teacher_id = ? AND subject = ?");
                $check->execute([$student['id'], $teacherId, $subject]);
                if (!$check->fetch()) {
                    $ins = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, teacher_notes) VALUES (?, ?, ?, 'Approved', 'Synced auto-link')");
                    $ins->execute([$student['id'], $teacherId, $subject]);
                    $syncedCount++;
                }
            }
        }

        echo json_encode([
            'success' => true,
            'message' => 'Penyelarasan berjaya.',
            'synced_count' => $syncedCount
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function autoAssignStudentToTeacher($studentId, $subject)
{
    global $pdo;

    if (empty($subject)) {
        return ['assigned' => false, 'subject' => 'Tiada', 'status' => 'Ralat Subjek'];
    }

    // Auto-assign based on SUBJECT ONLY - no capacity check
    // Any teacher who teaches this subject can take the student
    $stmt = $pdo->prepare("
        SELECT id, full_name
        FROM teachers
        WHERE JSON_CONTAINS(subjects, ?)
        AND status = 'Active'
        ORDER BY id ASC
        LIMIT 1
    ");
    $stmt->execute([json_encode($subject)]);
    $teacher = $stmt->fetch();

    if ($teacher) {
        // Found teacher - create enrollment with Pending status
        $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, auto_assigned) VALUES (?, ?, ?, 'Pending', 'Yes')");
        $stmt->execute([$studentId, $teacher['id'], $subject]);

        // Update student's enrolled_by if first enrollment
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM student_enrollments WHERE student_id = ? AND teacher_id = ?");
        $stmt->execute([$studentId, $teacher['id']]);
        if ($stmt->fetchColumn() == 1) {
            $stmt = $pdo->prepare("UPDATE students SET enrolled_by = ? WHERE id = ?");
            $stmt->execute([$teacher['id'], $studentId]);
        }

        return [
            'assigned' => true,
            'teacher_id' => $teacher['id'],
            'teacher_name' => $teacher['full_name'],
            'subject' => $subject,
            'status' => 'Menunggu Kelulusan'
        ];
    }

    // No teacher available at all for this subject
    // Create enrollment with NULL teacher_id for admin manual assignment
    $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, teacher_notes) VALUES (?, NULL, ?, 'Pending', 'Tiada guru tersedia - menunggu pentadbiran')");
    $stmt->execute([$studentId, $subject]);

    return [
        'assigned' => false,
        'subject' => $subject,
        'status' => 'Menunggu Guru (Tiada Available)'
    ];
}

/**
 * Get enrollments for a teacher (with status filter)
 * For multi-subject teachers, this shows all students across all subjects they teach
 */
function getTeacherEnrollments()
{
    global $pdo;
    requireTeacher();

    $status = isset($_GET['status']) ? $_GET['status'] : '';

    try {
        // Get all subjects this teacher teaches
        $teacherSubjects = json_decode($_SESSION['teacher_subjects'] ?? '[]', true);
        if (empty($teacherSubjects)) {
            $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $teacherSubjects = json_decode($stmt->fetchColumn() ?: '[]', true);
        }

        $query = "
            SELECT se.*, s.full_name as student_name, s.phone as student_phone, s.age, s.parent_guardian_name, s.grade_level
            FROM student_enrollments se
            JOIN students s ON se.student_id = s.id
            WHERE se.teacher_id = ?
        ";

        $params = [$_SESSION['user_id']];

        if (!empty($status) && $status !== 'all') {
            $query .= " AND se.status = ?";
            $params[] = $status;
        }

        $query .= " ORDER BY se.enrolled_at DESC";

        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $enrollments = $stmt->fetchAll();

        // Get stats
        $stmt = $pdo->prepare("SELECT status, COUNT(*) as count FROM student_enrollments WHERE teacher_id = ? GROUP BY status");
        $stmt->execute([$_SESSION['user_id']]);
        $stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        echo json_encode([
            'success' => true,
            'enrollments' => $enrollments,
            'stats' => [
                'pending' => $stats['Pending'] ?? 0,
                'approved' => $stats['Approved'] ?? 0,
                'rejected' => $stats['Rejected'] ?? 0
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Approve a student enrollment
 */
function approveStudent()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID enrollment tidak valid']);
        return;
    }

    try {
        // Update enrollment status
        $stmt = $pdo->prepare("UPDATE student_enrollments SET status = 'Approved', teacher_notes = ? WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$notes, $enrollmentId, $_SESSION['user_id']]);

        // Get student info for notification
        $stmt = $pdo->prepare("
            SELECT s.id, s.full_name, s.phone 
            FROM student_enrollments se 
            JOIN students s ON se.student_id = s.id 
            WHERE se.id = ?
        ");
        $stmt->execute([$enrollmentId]);
        $student = $stmt->fetch();

        if ($student) {
            createNotification($student['id'], 'student', 'Pendaftaran Diluluskan', 'Selamat! Pendaftaran anda untuk subjek telah diluluskan oleh guru.', 'success');
        }

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya diluluskan']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Reject a student enrollment with auto-fallback
 */
function rejectStudent()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $reason = isset($_POST['reason']) ? trim($_POST['reason']) : '';

    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID enrollment tidak valid']);
        return;
    }

    try {
        // Get current enrollment
        $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$enrollmentId, $_SESSION['user_id']]);
        $enrollment = $stmt->fetch();

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Enrollment tidak dijumpai']);
            return;
        }

        // Update status to rejected
        $stmt = $pdo->prepare("UPDATE student_enrollments SET status = 'Rejected', rejection_reason = ? WHERE id = ?");
        $stmt->execute([$reason, $enrollmentId]);

        // Get student info
        $stmt = $pdo->prepare("SELECT id, full_name, phone FROM students WHERE id = ?");
        $stmt->execute([$enrollment['student_id']]);
        $student = $stmt->fetch();

        // Notify student
        if ($student) {
            createNotification($student['id'], 'student', 'Pendaftaran Ditolak', 'Maaf, pendaftaran anda telah ditolak. Sila hubungi guru untuk maklumat lanjut.', 'error');
        }

        // Auto-fallback: Assign to next available teacher
        $nextTeacher = findNextAvailableTeacher($enrollment['subject'], $_SESSION['user_id']);

        if ($nextTeacher) {
            // Create new enrollment for next teacher
            $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status) VALUES (?, ?, ?, 'Pending')");
            $stmt->execute([$enrollment['student_id'], $nextTeacher['id'], $enrollment['subject']]);

            // Update teacher count
            $stmt = $pdo->prepare("UPDATE teacher_subjects SET current_students = current_students + 1 WHERE teacher_id = ? AND subject_name = ?");
            $stmt->execute([$nextTeacher['id'], $enrollment['subject']]);

            echo json_encode([
                'success' => true,
                'message' => 'Pelajar ditolak. Sistem telah auto-assign ke guru lain.',
                'fallback_assigned' => true,
                'next_teacher' => $nextTeacher['full_name']
            ]);
        } else {
            // Add to subject requests (waiting queue)
            $stmt = $pdo->prepare("INSERT INTO subject_requests (student_id, requested_subject, status) VALUES (?, ?, 'Pending')");
            $stmt->execute([$enrollment['student_id'], $enrollment['subject']]);

            echo json_encode([
                'success' => true,
                'message' => 'Pelajar ditolak. Tiada guru lain tersedia untuk subjek ini.',
                'fallback_assigned' => false,
                'waiting' => true
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Update enrollment information
 */
function updateEnrollment()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $studentName = isset($_POST['student_name']) ? trim($_POST['student_name']) : '';
    $studentPhone = isset($_POST['student_phone']) ? trim($_POST['student_phone']) : '';
    $subject = isset($_POST['subject']) ? trim($_POST['subject']) : '';

    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID enrollment tidak valid']);
        return;
    }

    try {
        // Verify enrollment belongs to this teacher
        $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$enrollmentId, $_SESSION['user_id']]);
        $enrollment = $stmt->fetch();

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Enrollment tidak dijumpai']);
            return;
        }

        // Update enrollment subject if changed
        if (!empty($subject) && $subject !== $enrollment['subject']) {
            $stmt = $pdo->prepare("UPDATE student_enrollments SET subject = ? WHERE id = ?");
            $stmt->execute([$subject, $enrollmentId]);
        }

        // Update student info in students table if provided
        if (!empty($studentName) || !empty($studentPhone)) {
            $updateFields = [];
            $updateParams = [];

            if (!empty($studentName)) {
                $updateFields[] = "full_name = ?";
                $updateParams[] = $studentName;
            }
            if (!empty($studentPhone)) {
                $updateFields[] = "phone = ?";
                $updateParams[] = $studentPhone;
            }

            if (!empty($updateFields)) {
                $updateParams[] = $enrollment['student_id'];
                $stmt = $pdo->prepare("UPDATE students SET " . implode(', ', $updateFields) . " WHERE id = ?");
                $stmt->execute($updateParams);
            }
        }

        echo json_encode([
            'success' => true,
            'message' => 'Maklumat pendaftaran berjaya dikemaskini'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Find next available teacher for a subject
 */
function findNextAvailableTeacher($subject, $excludeTeacherId = null)
{
    global $pdo;

    $query = "
        SELECT t.id, t.full_name, ts.current_students, ts.max_students
        FROM teachers t
        JOIN teacher_subjects ts ON t.id = ts.teacher_id
        WHERE ts.subject_name = ? 
        AND t.status = 'Active'
        AND ts.status = 'Active'
        AND ts.current_students < ts.max_students
    ";

    $params = [$subject];

    if ($excludeTeacherId) {
        $query .= " AND t.id != ?";
        $params[] = $excludeTeacherId;
    }

    $query .= " ORDER BY ts.current_students ASC LIMIT 1";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);

    return $stmt->fetch();
}

/**
 * Get enrollments for a student (all teachers)
 */
function getStudentEnrollments()
{
    global $pdo;
    requireStudent();

    try {
        $stmt = $pdo->prepare("
            SELECT se.*, t.full_name as teacher_name, t.phone as teacher_phone, t.profile_emoji as teacher_emoji
            FROM student_enrollments se
            JOIN teachers t ON se.teacher_id = t.id
            WHERE se.student_id = ?
            ORDER BY se.subject ASC, se.enrolled_at DESC
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $enrollments = $stmt->fetchAll();

        // Group by status
        $byStatus = [
            'approved' => [],
            'pending' => [],
            'rejected' => []
        ];

        foreach ($enrollments as $enrollment) {
            $status = strtolower($enrollment['status']);
            if (isset($byStatus[$status])) {
                $byStatus[$status][] = $enrollment;
            }
        }

        echo json_encode([
            'success' => true,
            'enrollments' => $enrollments,
            'by_status' => $byStatus
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Teacher registers a student directly (auto-approved)
 */
function teacherRegisterStudent()
{
    global $pdo;
    requireTeacher();

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $age = isset($_POST['age']) ? (int) $_POST['age'] : 0;
    $parentGuardian = isset($_POST['parent_guardian_name']) ? trim($_POST['parent_guardian_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';
    $gradeLevel = isset($_POST['grade_level']) ? $_POST['grade_level'] : '';
    $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : [];

    if (empty($fullName) || $age < 5 || $age > 25 || empty($parentGuardian) || empty($phone) || empty($educationLevel) || empty($gradeLevel) || empty($requiredSubjects)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan dengan betul']);
        return;
    }

    $username = generateUsername($fullName);
    $password = password_hash('pelajar', PASSWORD_DEFAULT);

    try {
        // Insert student
        $stmt = $pdo->prepare("INSERT INTO students (username, password, full_name, age, parent_guardian_name, phone, education_level, grade_level, required_subjects, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'Active')");
        $stmt->execute([$username, $password, $fullName, $age, $parentGuardian, $phone, $educationLevel, $gradeLevel, json_encode($requiredSubjects)]);

        $studentId = $pdo->lastInsertId();

        // Create auto-approved enrollments for this teacher
        foreach ($requiredSubjects as $subject) {
            $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, teacher_notes) VALUES (?, ?, ?, 'Approved', 'Didaftar oleh guru')");
            $stmt->execute([$studentId, $_SESSION['user_id'], $subject]);

            // Update teacher subject count
            $stmt = $pdo->prepare("UPDATE teacher_subjects SET current_students = current_students + 1 WHERE teacher_id = ? AND subject_name = ?");
            $stmt->execute([$_SESSION['user_id'], $subject]);
        }

        echo json_encode([
            'success' => true,
            'message' => 'Pelajar berjaya didaftarkan!',
            'username' => $username,
            'password' => 'pelajar'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get available teachers for a subject
 */
function getAvailableTeachers()
{
    global $pdo;

    $subject = isset($_GET['subject']) ? $_GET['subject'] : '';

    if (empty($subject)) {
        echo json_encode(['success' => false, 'message' => 'Subjek diperlukan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("
            SELECT t.id, t.full_name, t.phone, ts.current_students, ts.max_students
            FROM teachers t
            JOIN teacher_subjects ts ON t.id = ts.teacher_id
            WHERE ts.subject_name = ? 
            AND t.status = 'Active'
            AND ts.status = 'Active'
            AND ts.current_students < ts.max_students
            ORDER BY ts.current_students ASC
        ");
        $stmt->execute([$subject]);
        $teachers = $stmt->fetchAll();

        echo json_encode(['success' => true, 'teachers' => $teachers]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get subject requests (students waiting for teachers)
 */
function getSubjectRequests()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->prepare("
            SELECT sr.*, s.full_name as student_name, s.phone as student_phone
            FROM subject_requests sr
            JOIN students s ON sr.student_id = s.id
            WHERE sr.status = 'Pending'
            AND sr.requested_subject IN (
                SELECT subject_name FROM teacher_subjects WHERE teacher_id = ?
            )
            ORDER BY sr.created_at DESC
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $requests = $stmt->fetchAll();

        echo json_encode(['success' => true, 'requests' => $requests]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
function getAllSchedules()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || !in_array($_SESSION['user_type'], ['teacher', 'admin'])) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $teacherId = isset($_GET['teacher_id']) ? (int) $_GET['teacher_id'] : 0;
    $month = isset($_GET['month']) ? $_GET['month'] : '';
    $year = isset($_GET['year']) ? $_GET['year'] : date('Y');

    $query = "SELECT s.*, t.full_name as teacher_name, t.phone as teacher_phone, t.profile_emoji as teacher_emoji
              FROM schedule s
              JOIN teachers t ON s.teacher_id = t.id
              WHERE s.status = 'Active'";

    $params = [];

    if ($teacherId > 0) {
        $query .= " AND s.teacher_id = ?";
        $params[] = $teacherId;
    }

    if (!empty($month)) {
        $query .= " AND MONTH(s.schedule_date) = ?";
        $params[] = $month;
    }

    if (!empty($year)) {
        $query .= " AND YEAR(s.schedule_date) = ?";
        $params[] = $year;
    }

    $query .= " ORDER BY s.schedule_date, s.start_time";

    try {
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $schedules = $stmt->fetchAll();

        echo json_encode(['success' => true, 'schedules' => $schedules]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== ANNOUNCEMENT FUNCTIONS ==========

function getTeacherAnnouncements()
{
    global $pdo;
    requireTeacher();

    // Check if target_audience column exists and add if missing
    try {
        $check = $pdo->query("SHOW COLUMNS FROM announcements LIKE 'target_audience'");
        if ($check->rowCount() == 0) {
            $pdo->query("ALTER TABLE announcements ADD COLUMN target_audience VARCHAR(20) DEFAULT 'Semua' AFTER priority");
        }
    } catch (Exception $e) { /* Ignore */
    }

    try {
        $userId = $_SESSION['user_id'];

        // Show announcements targeted to Teachers + Created by Me
        $stmt = $pdo->prepare("
            SELECT a.*, t.full_name as teacher_name, t.profile_emoji,
                   CASE WHEN a.teacher_id = ? THEN 1 ELSE 0 END as is_owner
            FROM announcements a
            JOIN teachers t ON a.teacher_id = t.id
            WHERE (a.expiry_date IS NULL OR a.expiry_date >= CURDATE() OR a.expiry_date = '0000-00-00')
            AND (
                a.target_audience IN ('Semua', 'Guru Sahaja') 
                OR a.target_audience IS NULL
                OR a.teacher_id = ?
            )
            ORDER BY 
                CASE WHEN a.priority = 'Urgent' THEN 1 
                     WHEN a.priority = 'Penting' THEN 2 
                     ELSE 3 END,
                a.created_at DESC
        ");
        $stmt->execute([$userId, $userId]);
        $announcements = $stmt->fetchAll();

        echo json_encode(['success' => true, 'announcements' => $announcements]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function addAnnouncement()
{
    global $pdo;
    requireAdminOrTeacher();

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $category = isset($_POST['category']) ? $_POST['category'] : 'Umum';
    $priority = isset($_POST['priority']) ? $_POST['priority'] : 'Normal';
    $targetAudience = isset($_POST['target_audience']) ? $_POST['target_audience'] : 'Semua';
    $attachmentLink = isset($_POST['attachment_link']) ? trim($_POST['attachment_link']) : '';
    $expiryDate = isset($_POST['expiry_date']) && !empty($_POST['expiry_date']) ? $_POST['expiry_date'] : null;

    if (empty($title) || empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi tajuk dan kandungan']);
        return;
    }

    // Determine values
    $creatorId = ($_SESSION['user_type'] === 'admin') ? null : $_SESSION['user_id'];

    try {
        $stmt = $pdo->prepare("INSERT INTO announcements (teacher_id, title, content, category, priority, target_audience, attachment_link, expiry_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$creatorId, $title, $content, $category, $priority, $targetAudience, $attachmentLink, $expiryDate]);

        $announcementId = $pdo->lastInsertId();

        // Notify based on target
        if ($targetAudience == 'Semua' || $targetAudience == 'Guru Sahaja') {
            $currentUserId = $_SESSION['user_id'];
            $stmt = $pdo->query("SELECT id FROM teachers WHERE status = 'Active' AND id != $currentUserId");
            $teachers = $stmt->fetchAll(PDO::FETCH_COLUMN);
            foreach ($teachers as $teacherId) {
                createNotification($teacherId, 'teacher', 'Pengumuman Baru: ' . $title, 'Untuk Guru: ' . truncateText($content, 50), 'announcement');
            }
        }

        if ($targetAudience == 'Semua' || $targetAudience == 'Pelajar') {
            notifyAllStudents('Pengumuman Baru: ' . $title, 'Sila semak pengumuman terbaru.', 'announcement');
        }

        echo json_encode(['success' => true, 'message' => 'Pengumuman berjaya dihantar!', 'announcement_id' => $announcementId]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
function deleteAnnouncement()
{
    global $pdo;
    requireAdminOrTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak valid']);
        return;
    }

    try {
        if ($_SESSION['user_type'] === 'admin') {
            // Admin can delete ANY announcement
            $stmt = $pdo->prepare("DELETE FROM announcements WHERE id = ?");
            $stmt->execute([$id]);
        } else {
            // Teachers can ONLY delete their own
            $stmt = $pdo->prepare("DELETE FROM announcements WHERE id = ? AND teacher_id = ?");
            $stmt->execute([$id, $_SESSION['user_id']]);
        }

        echo json_encode(['success' => true, 'message' => 'Pengumuman berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateAnnouncement()
{
    global $pdo;
    requireAdminOrTeacher();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $priority = isset($_POST['priority']) ? $_POST['priority'] : 'Normal';
    $targetAudience = isset($_POST['target_audience']) ? $_POST['target_audience'] : 'Semua';

    if ($id === 0 || empty($title) || empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Data tidak lengkap']);
        return;
    }

    try {
        if ($_SESSION['user_type'] === 'admin') {
            // Admin can update ANY
            $stmt = $pdo->prepare("UPDATE announcements SET title = ?, content = ?, priority = ?, target_audience = ? WHERE id = ?");
            $stmt->execute([$title, $content, $priority, $targetAudience, $id]);
        } else {
            // Teachers can ONLY update their own
            $stmt = $pdo->prepare("UPDATE announcements SET title = ?, content = ?, priority = ?, target_audience = ? WHERE id = ? AND teacher_id = ?");
            $stmt->execute([$title, $content, $priority, $targetAudience, $id, $_SESSION['user_id']]);
        }

        if ($stmt->rowCount() > 0) {
            echo json_encode(['success' => true, 'message' => 'Pengumuman berjaya dikemaskini']);
        } else {
            // Check if it exists 
            $check = $pdo->prepare("SELECT id, teacher_id FROM announcements WHERE id = ?");
            $check->execute([$id]);
            $ann = $check->fetch();

            if (!$ann) {
                echo json_encode(['success' => false, 'message' => 'Pengumuman tidak dijumpai.']);
            } else if ($_SESSION['user_type'] === 'teacher' && $ann['teacher_id'] != $_SESSION['user_id']) {
                echo json_encode(['success' => false, 'message' => 'Anda hanya boleh mengemaskini pengumuman anda sendiri.']);
            } else {
                echo json_encode(['success' => true, 'message' => 'Tiada perubahan dibuat.']);
            }
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getLatestAnnouncement()
{
    global $pdo;

    try {
        $stmt = $pdo->query("
            SELECT a.*, t.full_name as teacher_name
            FROM announcements a
            JOIN teachers t ON a.teacher_id = t.id
            WHERE (a.expiry_date IS NULL OR a.expiry_date >= CURDATE())
            ORDER BY a.id DESC LIMIT 1
        ");
        $announcement = $stmt->fetch();

        echo json_encode(['success' => true, 'announcement' => $announcement]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== ADMIN ANNOUNCEMENT FUNCTIONS ==========

/**
 * Get all announcements for admin - including teacher announcements
 */
function getAdminAnnouncements()
{
    global $pdo;
    requireAdmin();

    try {
        $stmt = $pdo->query("
            SELECT a.*, t.full_name as teacher_name, t.profile_emoji
            FROM announcements a
            LEFT JOIN teachers t ON a.teacher_id = t.id
            ORDER BY 
                CASE WHEN a.priority = 'Urgent' THEN 1 
                     WHEN a.priority = 'Penting' THEN 2 
                     ELSE 3 END,
                a.created_at DESC
        ");
        $announcements = $stmt->fetchAll();

        // Calculate stats
        $total = count($announcements);
        $urgent = count(array_filter($announcements, function ($a) {
            return $a['priority'] === 'Urgent';
        }));
        $important = count(array_filter($announcements, function ($a) {
            return $a['priority'] === 'Penting';
        }));

        echo json_encode([
            'success' => true,
            'announcements' => $announcements,
            'stats' => [
                'total' => $total,
                'urgent' => $urgent,
                'important' => $important,
                'normal' => $total - $urgent - $important
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Admin creates announcement - notifies all teachers
 */
function addAnnouncementByAdmin()
{
    global $pdo;

    // Manual Admin Check
    if (!isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Sesi tamat. Sila log masuk semula.']);
        return;
    }
    // Debug Log
    error_log("Attempting to add admin announcement. User ID: " . $_SESSION['user_id']);

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';
    $content = isset($_POST['content']) ? trim($_POST['content']) : '';
    $category = isset($_POST['category']) ? $_POST['category'] : 'Umum';
    $priority = isset($_POST['priority']) ? $_POST['priority'] : 'Normal';
    $attachmentLink = isset($_POST['attachment_link']) ? trim($_POST['attachment_link']) : '';
    $expiryDate = isset($_POST['expiry_date']) && !empty($_POST['expiry_date']) ? $_POST['expiry_date'] : null;

    if (empty($title) || empty($content)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi tajuk dan kandungan pengumuman']);
        return;
    }

    // AUTO-FIX: Ensure table schema supports admin posts (Lazy Migration)
    try {
        // 1. Check if we need to modify teacher_id
        $pdo->query("ALTER TABLE announcements MODIFY teacher_id INT(11) NULL");
        // 2. Check/Add created_by
        $cols = $pdo->query("SHOW COLUMNS FROM announcements LIKE 'created_by'")->fetchAll();
        if (count($cols) == 0) {
            $pdo->query("ALTER TABLE announcements ADD COLUMN created_by INT(11) NULL AFTER teacher_id");
        }
    } catch (Exception $e) {
        // Ignore alter errors, table might already be fixed
    }

    // Insert announcement with admin as creator (teacher_id = NULL, created_by = admin_id)
    try {
        $stmt = $pdo->prepare("
                INSERT INTO announcements 
                (teacher_id, title, content, category, priority, attachment_link, expiry_date, created_by) 
                VALUES (NULL, ?, ?, ?, ?, ?, ?, ?)
            ");
        $stmt->execute([$title, $content, $category, $priority, $attachmentLink, $expiryDate, $_SESSION['user_id']]);

        $announcementId = $pdo->lastInsertId();

        // Get admin info for notification
        $adminName = $_SESSION['profile_name'] ?? 'Pentadbir';

        // Notify all active teachers
        $stmt = $pdo->query("SELECT id FROM teachers WHERE status = 'Active'");
        $teachers = $stmt->fetchAll(PDO::FETCH_COLUMN);
        foreach ($teachers as $teacherId) {
            createNotification($teacherId, 'teacher', '📢 Pengumuman Admin: ' . $title, $content, 'announcement');
        }

        // Also create notification for admin themselves
        createNotification($_SESSION['user_id'], 'admin', 'Pengumuman Dicipta', 'Anda telah mencipta pengumuman: ' . $title, 'success');

        echo json_encode([
            'success' => true,
            'message' => '✅ Pengumuman berjaya dihantar kepada semua guru!',
            'announcement_id' => $announcementId,
            'teachers_notified' => count($teachers)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Admin deletes announcement
 */
function deleteAnnouncementByAdmin()
{
    global $pdo;
    requireAdmin();

    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pengumuman tidak valid']);
        return;
    }

    try {
        // Get announcement info before deletion
        $stmt = $pdo->prepare("SELECT title FROM announcements WHERE id = ?");
        $stmt->execute([$id]);
        $announcement = $stmt->fetch();

        if (!$announcement) {
            echo json_encode(['success' => false, 'message' => 'Pengumuman tidak dijumpai']);
            return;
        }

        // Delete the announcement
        $stmt = $pdo->prepare("DELETE FROM announcements WHERE id = ?");
        $stmt->execute([$id]);

        echo json_encode([
            'success' => true,
            'message' => '✅ Pengumuman "' . $announcement['title'] . '" berjaya dipadam'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}
// ========== OPR SETTINGS FUNCTIONS ==========
function getOPRSettings()
{
    global $pdo;

    // Allow both admin and teacher to access OPR settings
    if (!isset($_SESSION['user_type']) || !in_array($_SESSION['user_type'], ['admin', 'teacher'])) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    try {
        // Get OPR settings from settings table
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'opr_info'");
        $stmt->execute();
        $oprData = $stmt->fetch();

        if ($oprData && $oprData['setting_value']) {
            $decoded = json_decode($oprData['setting_value'], true);
            if ($decoded) {
                echo json_encode(['success' => true, 'opr' => $decoded]);
            } else {
                // Fallback to default if JSON decode fails
                echo json_encode([
                    'success' => true,
                    'opr' => [
                        'title' => 'One Page Report (OPR)',
                        'description' => 'One Page Report merupakan dokumentasi ringkas yang menerangkan penggunaan One Page Report.',
                        'link' => 'https://www.canva.com/design/DAG6y7UmBKU/view'
                    ]
                ]);
            }
        } else {
            // Return default OPR info if no settings exist
            echo json_encode([
                'success' => true,
                'opr' => [
                    'title' => 'One Page Report (OPR)',
                    'description' => 'One Page Report merupakan dokumentasi ringkas yang menerangkan penggunaan One Page Report sebagai medium pelaporan utama dalam pengurusan.',
                    'link' => 'https://www.canva.com/design/DAG6y7UmBKU/view'
                ]
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function saveOprSettings()
{
    global $pdo;
    requireAdmin();
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $link = isset($_POST['link']) ? trim($_POST['link']) : '';
    if (empty($description) || empty($link)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi penerangan dan pautan']);
        return;
    }
    try {
        $oprInfo = json_encode(['title' => 'One Page Report (OPR)', 'description' => $description, 'link' => $link]);
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value, setting_type, description) VALUES ('opr_info', ?, 'json', 'One Page Report info and link') ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()");
        $stmt->execute([$oprInfo, $oprInfo]);
        echo json_encode(['success' => true, 'message' => 'Tetapan OPR berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== MAINTENANCE LOG FUNCTIONS ==========
function addMaintenanceLog()
{
    global $pdo;
    requireAdmin();
    $activity = isset($_POST['activity']) ? trim($_POST['activity']) : '';
    $status = isset($_POST['status']) ? $_POST['status'] : 'Aktif';
    $details = isset($_POST['details']) ? trim($_POST['details']) : '';
    if (empty($activity)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi aktiviti']);
        return;
    }
    try {
        $stmt = $pdo->prepare("INSERT INTO maintenance_logs (activity, status, details, created_by) VALUES (?, ?, ?, ?)");
        $stmt->execute([$activity, $status, $details, $_SESSION['user_id']]);
        echo json_encode(['success' => true, 'message' => 'Log penyelenggaraan berjaya ditambah']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getMaintenanceLogs()
{
    global $pdo;
    if (!isset($_SESSION['user_type']) || !in_array($_SESSION['user_type'], ['admin', 'teacher'])) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }
    try {
        $stmt = $pdo->query("SELECT * FROM maintenance_logs ORDER BY created_at DESC LIMIT 50");
        $logs = $stmt->fetchAll();
        echo json_encode(['success' => true, 'logs' => $logs]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function deleteMaintenanceLog()
{
    global $pdo;
    requireAdmin();
    $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak valid']);
        return;
    }
    try {
        $stmt = $pdo->prepare("DELETE FROM maintenance_logs WHERE id = ?");
        $stmt->execute([$id]);
        echo json_encode(['success' => true, 'message' => 'Log berjaya dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function updateMaintenanceLog()
{
    global $pdo;
    requireAdmin();
    $id = isset($_POST['log_id']) ? (int) $_POST['log_id'] : 0;
    $activity = isset($_POST['activity']) ? trim($_POST['activity']) : '';
    $status = isset($_POST['status']) ? $_POST['status'] : 'Aktif';
    $details = isset($_POST['details']) ? trim($_POST['details']) : '';

    if ($id === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak valid']);
        return;
    }

    if (empty($activity)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi aktiviti']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE maintenance_logs SET activity = ?, status = ?, details = ? WHERE id = ?");
        $stmt->execute([$activity, $status, $details, $id]);
        echo json_encode(['success' => true, 'message' => 'Log penyelenggaraan berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== FOOTER SETTINGS FUNCTIONS ==========
function saveFooterSettings()
{
    global $pdo;
    requireAdmin();
    $footerText = isset($_POST['footer_text']) ? trim($_POST['footer_text']) : 'Hak Cipta Terpelihara 2026';
    try {
        $stmt = $pdo->prepare("UPDATE admin SET footer_text = ? WHERE id = ?");
        $stmt->execute([$footerText, $_SESSION['user_id']]);
        echo json_encode(['success' => true, 'message' => 'Teks footer berjaya dikemaskini']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== STUDENT PASSWORD SETUP ==========
function setupStudentPassword()
{
    global $pdo;

    if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'teacher') {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';

    if ($studentId === 0 || empty($newPassword)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan']);
        return;
    }

    if (strlen($newPassword) < 6) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan minima 6 aksara']);
        return;
    }

    try {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE students SET password = ? WHERE id = ?");
        $stmt->execute([$hashedPassword, $studentId]);

        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        createNotification($studentId, 'student', 'Kata Laluan Dikemaskini', 'Guru telah menetapkan kata laluan baru untuk anda.', 'system');

        echo json_encode(['success' => true, 'message' => 'Kata laluan pelajar berjaya ditetapkan']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function changeStudentPassword()
{
    global $pdo;
    requireStudent();

    $currentPassword = isset($_POST['current_password']) ? $_POST['current_password'] : '';
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';

    if ($newPassword !== $confirmPassword) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan baru tidak sepadan']);
        return;
    }

    if (strlen($newPassword) < 6) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan minima 6 aksara']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT password FROM students WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $student = $stmt->fetch();

        if (!password_verify($currentPassword, $student['password'])) {
            echo json_encode(['success' => false, 'message' => 'Kata laluan semasa salah']);
            return;
        }

        $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE students SET password = ? WHERE id = ?");
        $stmt->execute([$newHash, $_SESSION['user_id']]);

        echo json_encode(['success' => true, 'message' => 'Kata laluan berjaya ditukar']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Register student with pending approval (auto password)
 * SIMPLIFIED VERSION - TIDAK GUNA student_enrollments table
 */
function registerStudentPending()
{
    global $pdo;

    $fullName = isset($_POST['full_name']) ? trim($_POST['full_name']) : '';
    $age = isset($_POST['age']) ? (int) $_POST['age'] : 0;
    $parentGuardian = isset($_POST['parent_guardian_name']) ? trim($_POST['parent_guardian_name']) : '';
    $phone = isset($_POST['phone']) ? trim($_POST['phone']) : '';
    $educationLevel = isset($_POST['education_level']) ? $_POST['education_level'] : '';
    $gradeLevel = isset($_POST['grade_level']) ? $_POST['grade_level'] : '';

    // Subjek
    $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : [];
    $customSubject = isset($_POST['custom_subject']) ? trim($_POST['custom_subject']) : '';
    if (!empty($customSubject)) {
        $requiredSubjects[] = $customSubject;
    }

    $enrolledBy = isset($_POST['enrolled_by']) ? (int) $_POST['enrolled_by'] : null;

    // Validasi
    if (empty($fullName) || $age < 5 || $age > 25 || empty($parentGuardian) || empty($phone) || empty($educationLevel) || empty($gradeLevel) || empty($requiredSubjects)) {
        echo json_encode(['success' => false, 'message' => 'Sila isi semua medan yang diperlukan dengan betul']);
        return;
    }

    // Generate unique username
    $username = generateUsername($fullName);

    // Default password (will be updated or set by teacher later)
    $password = password_hash('pelajar', PASSWORD_DEFAULT);

    try {
        // Insert student dengan status Pending (tunggu kelulusan guru untuk set password)
        $stmt = $pdo->prepare("INSERT INTO students (username, password, full_name, age, parent_guardian_name, phone, education_level, grade_level, required_subjects, enrolled_by, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')");

        $stmt->execute([$username, $password, $fullName, $age, $parentGuardian, $phone, $educationLevel, $gradeLevel, json_encode($requiredSubjects), $enrolledBy]);

        $studentId = $pdo->lastInsertId();

        echo json_encode([
            'success' => true,
            'title' => 'Pendaftaran Diterima! ⏳',
            'message' => "Pendaftaran anda telah diterima dan sedang diproses. Sila tunggu guru menetapkan kata laluan untuk anda.",
            'username' => $username,
            // Password not sent back as it's not set
        ]);

    } catch (Exception $e) {
        if ($e->getCode() == 23000) {
            echo json_encode(['success' => false, 'message' => 'Nama pelajar ini sudah didaftarkan.']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Ralat sistem: ' . $e->getMessage()]);
        }
    }
}

// ========== OPEN POOL LOGIC SUPPORT FUNCTIONS ==========

/**
 * Get Open Pool Students - using STUDENTS table ONLY
 * Shows all Active students who need approval for subjects teacher teaches
 * Teachers can see students based on their subject expertise
 */
function getOpenPoolStudents()
{
    global $pdo;
    requireTeacher();

    $subject = isset($_GET['subject']) ? trim($_GET['subject']) : '';

    try {
        // Get teacher's subjects from session or database
        $teacherSubjects = [];
        if (isset($_SESSION['teacher_subjects']) && !empty($_SESSION['teacher_subjects'])) {
            $teacherSubjects = json_decode($_SESSION['teacher_subjects'], true);
        }
        if (empty($teacherSubjects)) {
            $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $teacherSubjects = json_decode($stmt->fetchColumn() ?: '[]', true);
        }

        if (empty($teacherSubjects)) {
            echo json_encode(['success' => true, 'students' => [], 'message' => 'Tiada subjek didaftarkan', 'teacher_subjects' => []]);
            return;
        }

        // Get ALL Active students from students table ONLY
        // Then filter by required_subjects matching teacher's subjects
        $query = "
            SELECT 
                s.id as student_id,
                s.full_name as student_name,
                s.phone as student_phone,
                s.age,
                s.parent_guardian_name,
                s.education_level,
                s.grade_level,
                s.required_subjects,
                s.status as student_status,
                s.created_at as registered_at
            FROM students s
            WHERE s.status = 'Active'
        ";

        // Filter by subject if provided
        $params = [];
        if (!empty($subject)) {
            // Show students needing this specific subject
            $query .= " AND s.required_subjects LIKE ?";
            $params[] = '%' . $subject . '%';
        } else {
            // Show students needing ANY subject the teacher teaches
            $subjectConditions = [];
            foreach ($teacherSubjects as $idx => $subj) {
                $subjectConditions[] = "s.required_subjects LIKE ?";
                $params[] = '%' . $subj . '%';
            }
            if (!empty($subjectConditions)) {
                $query .= " AND (" . implode(' OR ', $subjectConditions) . ")";
            }
        }

        $query .= " ORDER BY s.created_at DESC";

        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $students = $stmt->fetchAll();

        // Process students - parse required_subjects and match with teacher subjects
        $processedStudents = [];
        foreach ($students as $student) {
            $studentSubjects = json_decode($student['required_subjects'], true) ?: [];
            $matchingSubjects = [];

            foreach ($studentSubjects as $subj) {
                // Check if teacher teaches this subject
                if (in_array($subj, $teacherSubjects)) {
                    $matchingSubjects[] = $subj;
                }
            }

            if (!empty($matchingSubjects)) {
                $processedStudent = [
                    'student_id' => $student['student_id'],
                    'student_name' => $student['student_name'],
                    'student_phone' => $student['student_phone'],
                    'age' => $student['age'],
                    'parent_guardian_name' => $student['parent_guardian_name'],
                    'education_level' => $student['education_level'],
                    'grade_level' => $student['grade_level'],
                    'required_subjects' => $student['required_subjects'],
                    'student_status' => $student['student_status'],
                    'registered_at' => $student['registered_at'],
                    'matching_subjects' => $matchingSubjects
                ];
                $processedStudents[] = $processedStudent;
            }
        }

        // Group by subject for easier display
        $bySubject = [];
        foreach ($processedStudents as $student) {
            foreach ($student['matching_subjects'] as $subj) {
                if (!isset($bySubject[$subj])) {
                    $bySubject[$subj] = [];
                }
                $studentCopy = $student;
                unset($studentCopy['matching_subjects']);
                $bySubject[$subj][] = $studentCopy;
            }
        }

        // Calculate stats per subject
        $stats = [];
        foreach ($bySubject as $subj => $studentsList) {
            $stats[$subj] = count($studentsList);
        }

        echo json_encode([
            'success' => true,
            'students' => $processedStudents,
            'by_subject' => $bySubject,
            'stats' => $stats,
            'teacher_subjects' => $teacherSubjects,
            'total_count' => count($processedStudents)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Approve student enrollment WITH password setup
 * This function ensures teacher MUST set student password before approval
 */
function approveStudentWithPassword()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

    // Validation
    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID enrollment tidak valid', 'require_password' => true]);
        return;
    }

    if ($studentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar tidak valid', 'require_password' => true]);
        return;
    }

    if (empty($password)) {
        echo json_encode(['success' => false, 'message' => '⚠️ WAJIB: Sila tetapkan kata laluan pelajar terlebih dahulu!', 'require_password' => true]);
        return;
    }

    if (strlen($password) < 6) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan minima 6 aksara', 'require_password' => true]);
        return;
    }

    try {
        // Verify enrollment belongs to this teacher
        $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$enrollmentId, $_SESSION['user_id']]);
        $enrollment = $stmt->fetch();

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Enrollment tidak dijumpai']);
            return;
        }

        // Hash and set student password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE students SET password = ?, password_set = 'Yes' WHERE id = ?");
        $stmt->execute([$hashedPassword, $studentId]);

        // Update enrollment status to Approved
        $stmt = $pdo->prepare("UPDATE student_enrollments SET status = 'Approved', teacher_notes = ? WHERE id = ?");
        $stmt->execute([$notes ?: 'Diluluskan dengan kata laluan ditetapkan', $enrollmentId]);

        // Update teacher subject count
        $stmt = $pdo->prepare("UPDATE teacher_subjects SET current_students = current_students + 1 WHERE teacher_id = ? AND subject_name = ?");
        $stmt->execute([$_SESSION['user_id'], $enrollment['subject']]);

        // Get student info for notification
        $stmt = $pdo->prepare("SELECT full_name, phone FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if ($student) {
            createNotification(
                $studentId,
                'student',
                '✅ Pendaftaran Diluluskan!',
                "Selamat! Pendaftaran anda untuk subjek {$enrollment['subject']} telah diluluskan.\n\nKata laluan baharu telah ditetapkan oleh guru.\nSila log masuk menggunakan kata laluan baharu.",
                'success'
            );
        }

        echo json_encode([
            'success' => true,
            'message' => "✅ Pelajar berjaya diluluskan!\n\nKata laluan telah ditetapkan untuk {$student['full_name']}.",
            'student_name' => $student['full_name'] ?? ''
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Reject student enrollment WITH auto-fallback to other teachers
 */
function rejectStudentWithFallback()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $reason = isset($_POST['reason']) ? trim($_POST['reason']) : '';
    $rejectType = isset($_POST['reject_type']) ? $_POST['reject_type'] : 'auto'; // 'auto' or 'manual'

    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID enrollment tidak valid']);
        return;
    }

    if (empty($reason)) {
        echo json_encode(['success' => false, 'message' => 'Sila nyatakan sebab penolakan']);
        return;
    }

    try {
        // Get current enrollment
        $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$enrollmentId, $_SESSION['user_id']]);
        $enrollment = $stmt->fetch();

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Enrollment tidak dijumpai']);
            return;
        }

        $studentId = $enrollment['student_id'];
        $subject = $enrollment['subject'];

        // Update current enrollment to Rejected
        $stmt = $pdo->prepare("UPDATE student_enrollments SET status = 'Rejected', rejection_reason = ? WHERE id = ?");
        $stmt->execute([$reason, $enrollmentId]);

        // Get student info
        $stmt = $pdo->prepare("SELECT full_name, phone FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        // Notify student about rejection
        if ($student) {
            $notifyMsg = "Maaf, pendaftaran anda untuk subjek {$subject} telah ditolak.\n";
            $notifyMsg .= "Sebab: {$reason}\n\n";

            if ($rejectType === 'auto') {
                $notifyMsg .= "Sistem sedang mencari guru lain untuk anda...";
            } else {
                $notifyMsg .= "Sila hubungi pentadbiran untuk maklumat lanjut.";
            }

            createNotification($studentId, 'student', '❌ Pendaftaran Ditolak', $notifyMsg, 'error');
        }

        if ($rejectType === 'auto') {
            // Auto-fallback: Find next available teacher
            $nextTeacher = findNextAvailableTeacher($subject, $_SESSION['user_id']);

            if ($nextTeacher) {
                // Create new enrollment for next teacher
                $stmt = $pdo->prepare("
                    INSERT INTO student_enrollments 
                    (student_id, teacher_id, subject, status, teacher_notes, auto_assigned) 
                    VALUES (?, ?, ?, 'Pending', 'Auto-assign selepas penolakan oleh guru lain', 'Yes')
                ");
                $stmt->execute([$studentId, $nextTeacher['id'], $subject]);

                // Update teacher count
                $stmt = $pdo->prepare("UPDATE teacher_subjects SET current_students = current_students + 1 WHERE teacher_id = ? AND subject_name = ?");
                $stmt->execute([$nextTeacher['id'], $subject]);

                // Notify the new teacher
                createNotification(
                    $nextTeacher['id'],
                    'teacher',
                    '📋 Auto-Assign: Pelajar Baru',
                    "Pelajar {$student['full_name']} telah di-assign secara automatik kepada anda untuk subjek {$subject}.\n\nSila buat semakan dan tetapkan kata laluan pelajar.",
                    'enrollment'
                );

                echo json_encode([
                    'success' => true,
                    'message' => "✅ Pendaftaran ditolak.\n\n📋 Pelajar telah di-assign automatik kepada guru lain:\n{$nextTeacher['full_name']}",
                    'fallback_assigned' => true,
                    'next_teacher' => $nextTeacher['full_name']
                ]);
            } else {
                // No other teacher available - add to subject_requests
                $stmt = $pdo->prepare("
                    INSERT INTO subject_requests (student_id, requested_subject, status, rejection_reason) 
                    VALUES (?, ?, 'Pending', ?)
                ");
                $stmt->execute([$studentId, $subject, $reason]);

                echo json_encode([
                    'success' => true,
                    'message' => "✅ Pendaftaran ditolak.\n\n⚠️ Tiada guru lain tersedia untuk subjek {$subject}.\nPelajar telah masuk dalam senarai menunggu.",
                    'fallback_assigned' => false,
                    'waiting' => true
                ]);
            }
        } else {
            // Manual rejection - no fallback
            echo json_encode([
                'success' => true,
                'message' => "✅ Pendaftaran ditolak.\n\nSebab: {$reason}",
                'fallback_assigned' => false
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get available teachers by subject for manual assignment
 */
function getAvailableTeachersBySubject()
{
    global $pdo;

    $subject = isset($_GET['subject']) ? trim($_GET['subject']) : '';

    if (empty($subject)) {
        echo json_encode(['success' => false, 'message' => 'Subjek diperlukan']);
        return;
    }

    try {
        $stmt = $pdo->prepare("
            SELECT 
                t.id,
                t.full_name,
                t.phone,
                t.profile_emoji,
                ts.current_students,
                ts.max_students,
                (ts.max_students - ts.current_students) as available_slots
            FROM teachers t
            JOIN teacher_subjects ts ON t.id = ts.teacher_id
            WHERE ts.subject_name = ? 
            AND t.status = 'Active'
            AND ts.status = 'Active'
            AND t.id != ?
            ORDER BY available_slots DESC, t.full_name ASC
        ");
        $stmt->execute([$subject, $_SESSION['user_id'] ?? 0]);
        $teachers = $stmt->fetchAll();

        // Also get current teacher info
        if (isset($_SESSION['user_id']) && $_SESSION['user_type'] === 'teacher') {
            $stmt = $pdo->prepare("
                SELECT 
                    t.id,
                    t.full_name,
                    ts.current_students,
                    ts.max_students,
                    (ts.max_students - ts.current_students) as available_slots
                FROM teachers t
                JOIN teacher_subjects ts ON t.id = ts.teacher_id
                WHERE t.id = ? AND ts.subject_name = ?
            ");
            $stmt->execute([$_SESSION['user_id'], $subject]);
            $currentTeacher = $stmt->fetch();
        }

        echo json_encode([
            'success' => true,
            'teachers' => $teachers,
            'current_teacher' => $currentTeacher ?? null
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Manually assign a student to another teacher
 */
function manualAssignStudent()
{
    global $pdo;
    requireTeacher();

    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;
    $subject = isset($_POST['subject']) ? trim($_POST['subject']) : '';
    $targetTeacherId = isset($_POST['target_teacher_id']) ? (int) $_POST['target_teacher_id'] : 0;
    $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

    if ($studentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID pelajar tidak valid']);
        return;
    }

    if (empty($subject)) {
        echo json_encode(['success' => false, 'message' => 'Subjek diperlukan']);
        return;
    }

    if ($targetTeacherId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID guru sasaran diperlukan']);
        return;
    }

    try {
        // Verify target teacher exists and teaches this subject
        $stmt = $pdo->prepare("
            SELECT t.*, ts.max_students, ts.current_students
            FROM teachers t
            JOIN teacher_subjects ts ON t.id = ts.teacher_id
            WHERE t.id = ? AND ts.subject_name = ? AND t.status = 'Active' AND ts.status = 'Active'
        ");
        $stmt->execute([$targetTeacherId, $subject]);
        $targetTeacher = $stmt->fetch();

        if (!$targetTeacher) {
            echo json_encode(['success' => false, 'message' => 'Guru sasaran tidak mengajar subjek ini atau tidak aktif']);
            return;
        }

        if ($targetTeacher['current_students'] >= $targetTeacher['max_students']) {
            echo json_encode(['success' => false, 'message' => 'Guru sasaran telah mencapai kapasiti maksimum']);
            return;
        }

        // Get student info
        $stmt = $pdo->prepare("SELECT full_name, phone FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if (!$student) {
            echo json_encode(['success' => false, 'message' => 'Pelajar tidak dijumpai']);
            return;
        }

        // Create new enrollment for target teacher
        $stmt = $pdo->prepare("
            INSERT INTO student_enrollments 
            (student_id, teacher_id, subject, status, teacher_notes, auto_assigned) 
            VALUES (?, ?, ?, 'Pending', ?, 'No')
        ");
        $stmt->execute([$studentId, $targetTeacherId, $subject, $notes ?: 'Dassign secara manual oleh guru lain']);

        // Update teacher subject count
        $stmt = $pdo->prepare("UPDATE teacher_subjects SET current_students = current_students + 1 WHERE teacher_id = ? AND subject_name = ?");
        $stmt->execute([$targetTeacherId, $subject]);

        // Notify the target teacher
        createNotification(
            $targetTeacherId,
            'teacher',
            '👨‍🎓 Pelajar Baharu Di-assign',
            "Pelajar {$student['full_name']} telah di-assign kepada anda untuk subjek {$subject}.\n\nSila buat semakan dan tetapkan kata laluan pelajar.",
            'enrollment'
        );

        // Notify student
        createNotification(
            $studentId,
            'student',
            '📋 Guru Baru Dassign',
            "Sistem telah assign guru baharu untuk subjek {$subject}: {$targetTeacher['full_name']}.\n\nSila tunggu kelulusan daripada guru baharu.",
            'info'
        );

        echo json_encode([
            'success' => true,
            'message' => "✅ Pelajar berjaya di-assign kepada {$targetTeacher['full_name']}",
            'teacher_name' => $targetTeacher['full_name']
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get students who have been approved but password not yet set
 * This helps teachers know which students need password setup
 */
function getPendingPasswordStudents()
{
    global $pdo;
    requireTeacher();

    try {
        // Get students who are approved but password not set
        // Students approved by this teacher who haven't had password set
        $stmt = $pdo->prepare("
            SELECT DISTINCT 
                s.id as student_id,
                s.full_name as student_name,
                s.phone as student_phone,
                s.education_level,
                s.grade_level,
                se.subject,
                se.status as enrollment_status,
                se.enrolled_at,
                se.approved_at,
                (s.password_set = 'No' OR s.password_set IS NULL OR s.password_set = '') as needs_password
            FROM students s
            JOIN student_enrollments se ON s.id = se.student_id
            WHERE se.teacher_id = ?
            AND se.status = 'Approved'
            AND s.status = 'Active'
            AND (s.password_set = 'No' OR s.password_set IS NULL OR s.password_set = '')
            ORDER BY se.approved_at DESC
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $students = $stmt->fetchAll();

        // Count by subject
        $bySubject = [];
        foreach ($students as $student) {
            $subj = $student['subject'];
            if (!isset($bySubject[$subj])) {
                $bySubject[$subj] = 0;
            }
            $bySubject[$subj]++;
        }

        echo json_encode([
            'success' => true,
            'students' => $students,
            'by_subject' => $bySubject,
            'total' => count($students)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== END OF API FUNCTIONS ==========
// No closing tag needed for pure PHP files

/**
 * Get Open Pool statistics
 */
function getOpenPoolStats()
{
    global $pdo;
    requireTeacher();

    try {
        // Get teacher subjects
        $teacherId = $_SESSION['user_id'];
        $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
        $stmt->execute([$teacherId]);
        $teacher = $stmt->fetch();
        $teacherSubjects = json_decode($teacher['subjects'] ?? '[]', true) ?: [];

        // Count pending students for teacher's subjects
        $pendingQuery = "
            SELECT COUNT(DISTINCT s.id)
            FROM students s
            LEFT JOIN student_enrollments se ON s.id = se.student_id
            WHERE s.status = 'Active'
            AND (
                se.id IS NULL
                OR (
                    se.subject IN ('" . implode("','", array_map(function ($s) use ($pdo) {
            return $pdo->quote($s);
        }, $teacherSubjects)) . "')
                    AND se.status = 'Rejected'
                    AND NOT EXISTS (
                        SELECT 1 FROM student_enrollments se2 
                        WHERE se2.student_id = s.id 
                        AND se2.status = 'Approved'
                        AND se2.subject IN ('" . implode("','", array_map(function ($s) use ($pdo) {
            return $pdo->quote($s);
        }, $teacherSubjects)) . "')
                    )
                )
            )
        ";
        $stmt = $pdo->prepare($pendingQuery);
        $stmt->execute();
        $pending = $stmt->fetchColumn();

        // Count approved today
        $today = date('Y-m-d');
        $approvedTodayQuery = "
            SELECT COUNT(DISTINCT s.id)
            FROM students s
            JOIN student_enrollments se ON s.id = se.student_id
            WHERE se.teacher_id = ?
            AND se.status = 'Approved'
            AND DATE(se.approved_at) = ?
        ";
        $stmt = $pdo->prepare($approvedTodayQuery);
        $stmt->execute([$teacherId, $today]);
        $approvedToday = $stmt->fetchColumn();

        echo json_encode([
            'success' => true,
            'stats' => [
                'pending' => (int) $pending,
                'approved_today' => (int) $approvedToday,
                'approved' => 0,
                'rejected' => 0,
                'total' => (int) $pending
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Approve a student from Open Pool and set password
 */
function approveOpenPoolStudent()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;
    $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
    $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

    if ($enrollmentId === 0 && $studentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak valid']);
        return;
    }

    if (empty($newPassword) || strlen($newPassword) < 6) {
        echo json_encode(['success' => false, 'message' => 'Kata laluan minima 6 aksara']);
        return;
    }

    try {
        // Get enrollment info
        if ($enrollmentId > 0) {
            $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ?");
            $stmt->execute([$enrollmentId]);
            $enrollment = $stmt->fetch();
        } else {
            $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE student_id = ? AND status = 'Pending' ORDER BY id DESC LIMIT 1");
            $stmt->execute([$studentId]);
            $enrollment = $stmt->fetch();
        }

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Pendaftaran tidak dijumpai']);
            return;
        }

        $studentId = $enrollment['student_id'];
        $subject = $enrollment['subject'];
        $teacherId = $_SESSION['user_id'];

        // Check if teacher teaches this subject
        $stmt = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
        $stmt->execute([$teacherId]);
        $teacher = $stmt->fetch();
        $teacherSubjects = json_decode($teacher['subjects'] ?? '[]', true) ?: [];

        if (!in_array($subject, $teacherSubjects)) {
            echo json_encode(['success' => false, 'message' => 'Anda tidak mengajar subjek ini']);
            return;
        }

        // Get student info
        $stmt = $pdo->prepare("SELECT full_name, phone FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if (!$student) {
            echo json_encode(['success' => false, 'message' => 'Pelajar tidak dijumpai']);
            return;
        }

        // Update student password
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE students SET password = ?, password_set = 'Yes', status = 'Active' WHERE id = ?");
        $stmt->execute([$hashedPassword, $studentId]);

        // Update enrollment status
        $stmt = $pdo->prepare("
            UPDATE student_enrollments 
            SET status = 'Approved', 
                teacher_id = ?, 
                approved_at = NOW(),
                teacher_notes = ?
            WHERE id = ?
        ");
        $stmt->execute([$teacherId, $notes ?: 'Diluluskan dari Open Pool', $enrollmentId]);

        // Update teacher subject count
        $stmt = $pdo->prepare("
            UPDATE teacher_subjects 
            SET current_students = current_students + 1 
            WHERE teacher_id = ? AND subject_name = ?
        ");
        $stmt->execute([$teacherId, $subject]);

        // Notify student
        createNotification(
            $studentId,
            'student',
            '✅ Pendaftaran Diluluskan',
            "Selamat! Pendaftaran anda untuk subjek {$subject} telah diluluskan oleh guru.\n\nAnda kini boleh log masuk dengan kata laluan baharu.",
            'success'
        );

        echo json_encode([
            'success' => true,
            'message' => '✅ Pelajar berjaya diluluskan dan kata laluan ditetapkan!'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Reject student with auto-fallback (assign to other teacher)
 */
function rejectWithAutoFallback()
{
    global $pdo;
    requireTeacher();

    $enrollmentId = isset($_POST['enrollment_id']) ? (int) $_POST['enrollment_id'] : 0;
    $reason = isset($_POST['reason']) ? trim($_POST['reason']) : '';

    if ($enrollmentId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak valid']);
        return;
    }

    if (empty($reason)) {
        echo json_encode(['success' => false, 'message' => 'Sebab penolakan diperlukan']);
        return;
    }

    try {
        // Get enrollment info
        $stmt = $pdo->prepare("SELECT * FROM student_enrollments WHERE id = ?");
        $stmt->execute([$enrollmentId]);
        $enrollment = $stmt->fetch();

        if (!$enrollment) {
            echo json_encode(['success' => false, 'message' => 'Pendaftaran tidak dijumpai']);
            return;
        }

        $studentId = $enrollment['student_id'];
        $subject = $enrollment['subject'];
        $rejectingTeacherId = $_SESSION['user_id'];

        // Update current enrollment as rejected
        $stmt = $pdo->prepare("
            UPDATE student_enrollments 
            SET status = 'Rejected', 
                rejected_reason = ?,
                rejected_by = ?,
                rejected_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute(["[Auto-Fallback] {$reason}", $rejectingTeacherId, $enrollmentId]);

        // Get student info
        $stmt = $pdo->prepare("SELECT full_name, phone, parent_guardian_name FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        // Find other teachers who teach this subject and have capacity
        $stmt = $pdo->prepare("
            SELECT t.id, t.full_name, ts.current_students, ts.max_students
            FROM teachers t
            JOIN teacher_subjects ts ON t.id = ts.teacher_id
            WHERE ts.subject_name = ?
            AND t.status = 'Active'
            AND ts.status = 'Active'
            AND t.id != ?
            AND ts.current_students < ts.max_students
            ORDER BY ts.current_students ASC, t.full_name ASC
            LIMIT 1
        ");
        $stmt->execute([$subject, $rejectingTeacherId]);
        $otherTeacher = $stmt->fetch();

        if ($otherTeacher) {
            // Assign to another teacher
            $stmt = $pdo->prepare("
                INSERT INTO student_enrollments
                (student_id, teacher_id, subject, status, teacher_notes, auto_assigned, created_at)
                VALUES (?, ?, ?, 'Pending', ?, 'Yes', NOW())
            ");
            $stmt->execute([
                $studentId,
                $otherTeacher['id'],
                $subject,
                "Auto-assign dari Open Pool (Guru sebelumnya menolak: {$reason})"
            ]);

            // Notify the new teacher
            createNotification(
                $otherTeacher['id'],
                'teacher',
                '👨‍🎓 Pelajar Baharu (Auto-Assign)',
                "Sistem telah auto-assign pelajar baharu kepada anda:\n\n" .
                "Nama: {$student['full_name']}\n" .
                "Subjek: {$subject}\n" .
                "No. Telefon: {$student['phone']}\n\n" .
                "Sila buat semakan dan tetapkan kata laluan pelajar.",
                'enrollment'
            );

            // Notify student
            createNotification(
                $studentId,
                'student',
                '📋 Guru Baru Dassign',
                "Maaf, guru sebelumnya tidak dapat menerima pendaftaran anda.\n\n" .
                "Sistem telah assign guru baharu untuk subjek {$subject}. Sila tunggu kelulusan daripada guru baharu.",
                'info'
            );

            echo json_encode([
                'success' => true,
                'message' => "✅ Pendaftaran ditolak dan pelajar telah di-assign ke {$otherTeacher['full_name']}",
                'assigned_teacher' => $otherTeacher['full_name']
            ]);
        } else {
            // No teacher available, keep in open pool
            echo json_encode([
                'success' => true,
                'message' => "Pendaftaran ditolak. Tiada guru lain tersedia untuk subjek ini. Pelajar kekal dalam Open Pool.",
                'assigned_teacher' => null
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

// ========== STATUS MANAGEMENT FUNCTIONS ==========

/**
 * Update teacher status (Active/Inactive)
 * Teachers can set themselves to Inactive (e.g., for leave)
 */
function updateTeacherStatus()
{
    global $pdo;
    requireTeacher();

    $status = isset($_POST['status']) ? $_POST['status'] : 'Active';

    if (!in_array($status, ['Active', 'Inactive'])) {
        echo json_encode(['success' => false, 'message' => 'Status tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE teachers SET status = ? WHERE id = ?");
        $stmt->execute([$status, $_SESSION['user_id']]);

        $statusText = $status === 'Active' ? 'Aktif' : 'Tidak Aktif';
        echo json_encode([
            'success' => true,
            'message' => "Status berjaya dikemaskini kepada: $statusText",
            'status' => $status
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Update admin status (Active/Inactive)
 * Admin can set themselves to Inactive
 */
function updateAdminStatus()
{
    global $pdo;
    requireAdmin();

    $status = isset($_POST['status']) ? $_POST['status'] : 'Active';

    if (!in_array($status, ['Active', 'Inactive'])) {
        echo json_encode(['success' => false, 'message' => 'Status tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE admin SET status = ? WHERE id = ?");
        $stmt->execute([$status, $_SESSION['user_id']]);

        $statusText = $status === 'Active' ? 'Aktif' : 'Tidak Aktif';
        echo json_encode([
            'success' => true,
            'message' => "Status berjaya dikemaskini kepada: $statusText",
            'status' => $status
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Update teacher status by admin
 * Admin can set teacher to Active or Inactive
 */
function updateTeacherStatusByAdmin()
{
    global $pdo;
    requireAdmin();

    $teacherId = isset($_POST['teacher_id']) ? (int) $_POST['teacher_id'] : 0;
    $status = isset($_POST['status']) ? $_POST['status'] : 'Active';

    if ($teacherId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID guru tidak valid']);
        return;
    }

    if (!in_array($status, ['Active', 'Inactive'])) {
        echo json_encode(['success' => false, 'message' => 'Status tidak valid']);
        return;
    }

    try {
        // Get teacher info for notification
        $stmt = $pdo->prepare("SELECT full_name FROM teachers WHERE id = ?");
        $stmt->execute([$teacherId]);
        $teacher = $stmt->fetch();

        if (!$teacher) {
            echo json_encode(['success' => false, 'message' => 'Guru tidak dijumpai']);
            return;
        }

        // Update teacher status
        $stmt = $pdo->prepare("UPDATE teachers SET status = ? WHERE id = ?");
        $stmt->execute([$status, $teacherId]);

        $statusText = $status === 'Active' ? 'Aktif' : 'Tidak Aktif';

        // Create notification for teacher
        $notificationTitle = $status === 'Active' ? 'Akaun Diaktifkan' : 'Akaun Dinyahaktifkan';
        $notificationMessage = $status === 'Active'
            ? 'Pentadbir telah mengaktifkan akaun anda. Anda kini boleh log masuk dan menerima pelajar.'
            : 'Pentadbir telah menyahaktifkan akaun anda. Anda tidak akan menerima pelajar baharu.';

        createNotification($teacherId, 'teacher', $notificationTitle, $notificationMessage, 'info');

        echo json_encode([
            'success' => true,
            'message' => "Status guru <strong>{$teacher['full_name']}</strong> telah dikemaskini kepada: <strong>$statusText</strong>",
            'status' => $status,
            'teacher_name' => $teacher['full_name']
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Reset teacher password by admin
 * Admin can reset teacher password to default "tutor123"
 */
function resetTeacherPassword()
{
    global $pdo;
    requireAdmin();

    $username = isset($_POST['username']) ? trim($_POST['username']) : '';

    if (empty($username)) {
        echo json_encode(['success' => false, 'message' => 'Username diperlukan']);
        return;
    }

    try {
        // Get teacher info
        $stmt = $pdo->prepare("SELECT id, full_name FROM teachers WHERE username = ?");
        $stmt->execute([$username]);
        $teacher = $stmt->fetch();

        if (!$teacher) {
            echo json_encode(['success' => false, 'message' => 'Guru tidak dijumpai dengan username: ' . $username]);
            return;
        }

        // Reset password to default "tutor123"
        $defaultPassword = password_hash('tutor123', PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE teachers SET password = ? WHERE id = ?");
        $stmt->execute([$defaultPassword, $teacher['id']]);

        // Notify the teacher
        createNotification(
            $teacher['id'],
            'teacher',
            'Kata Laluan Ditetapkan Semula',
            'Pentadbir telah menetapkan semula kata laluan akaun anda kepada: tutor123',
            'info'
        );

        echo json_encode([
            'success' => true,
            'message' => '✅ Kata laluan guru <strong>' . $teacher['full_name'] . '</strong> telah ditetapkan kepada: <code>tutor123</code>'
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}


/**
 * Get combined Student Management Data for Teacher Dashboard
 * Returns: active_records (Part I), new_applications (Part II)
 */
function getStudentManagementData()
{
    global $pdo;
    requireTeacher();

    try {
        // Schema Check (Auto-migration for enrolled_by)
        try {
            $colCheck = $pdo->query("SHOW COLUMNS FROM students LIKE 'enrolled_by'");
            if (!$colCheck->fetch()) {
                $pdo->exec("ALTER TABLE students ADD COLUMN enrolled_by INT(11) DEFAULT NULL AFTER status");
            }
        } catch (Exception $ex) { /* Ignore if fails, might be permissions */
        }

        $teacherId = $_SESSION['user_id'];

        // 1. BAHAGIAN I: Rekod Pelajar Aktif (Student yang status Active AND password IS NOT NULL)
        // Note: We assume 'Active' implies password is set, based on our new flow.
        // 1. BAHAGIAN I: Rekod Pelajar Aktif (Student yang status Active AND password IS NOT NULL)
        $stmt = $pdo->query("
            SELECT id, username, full_name, age, phone, education_level, grade_level, 
                   required_subjects, profile_emoji, status, created_at, parent_guardian_name, enrolled_by
            FROM students
            WHERE status = 'Active'
            ORDER BY full_name ASC
        ");
        $activeRecords = $stmt->fetchAll();

        // 2. BAHAGIAN II: Permohonan Baru (Student yang status Pending)
        // Pendaftaran dari register.php masuk sebagai Pending.
        $stmt = $pdo->query("
            SELECT id, username, full_name, age, phone, education_level, grade_level, 
                   required_subjects, profile_emoji, status, created_at, parent_guardian_name
            FROM students
            WHERE status = 'Pending'
            ORDER BY created_at DESC
        ");
        $newApplications = $stmt->fetchAll();

        // 3. BAHAGIAN III: Pelajar Saya (Primary Mentor OR Subject Teacher)
        $stmt = $pdo->prepare("
            SELECT s.id, s.username, s.full_name, s.age, s.phone, s.education_level, s.grade_level, 
                   s.required_subjects, s.profile_emoji, s.status, s.created_at, s.parent_guardian_name, s.enrolled_by,
                   NULL as assigned_subject
            FROM students s
            WHERE s.status = 'Active' AND s.enrolled_by = ?
            
            UNION
            
            SELECT s.id, s.username, s.full_name, s.age, s.phone, s.education_level, s.grade_level, 
                   s.required_subjects, s.profile_emoji, s.status, s.created_at, s.parent_guardian_name, s.enrolled_by,
                   se.subject as assigned_subject
            FROM students s
            JOIN student_enrollments se ON s.id = se.student_id
            WHERE s.status = 'Active' AND se.teacher_id = ? AND se.status = 'Approved'
            
            ORDER BY full_name ASC
        ");
        $stmt->execute([$teacherId, $teacherId]);
        $rawMyStudents = $stmt->fetchAll();

        // DE-DUPLICATE: Proccess the list to ensure each student appears once
        $myStudentsMap = [];
        foreach ($rawMyStudents as $s) {
            $id = $s['id'];
            if (!isset($myStudentsMap[$id])) {
                $myStudentsMap[$id] = $s;
            } else {
                // If we find a version with a specific subject, prefer it over the one without
                if (empty($myStudentsMap[$id]['assigned_subject']) && !empty($s['assigned_subject'])) {
                    $myStudentsMap[$id]['assigned_subject'] = $s['assigned_subject'];
                }
            }
        }
        $myStudents = array_values($myStudentsMap);

        echo json_encode([
            'success' => true,
            'teacher_id' => $teacherId,
            'active_records' => $activeRecords,
            'new_applications' => $newApplications,
            'my_students' => $myStudents
        ]);

    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}

function approveStudentApplication()
{
    global $pdo;
    requireTeacher();

    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $useDefault = isset($_POST['use_default']) && $_POST['use_default'] == 'true';

    if ($studentId <= 0) {
        echo json_encode(['success' => false, 'message' => 'ID Pelajar tidak sah']);
        return;
    }

    if ($useDefault) {
        $password = 'pelajar';
    }

    if (empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Sila tetapkan kata laluan']);
        return;
    }

    try {
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        $teacherId = $_SESSION['user_id'];

        // 1. Update status AND set enrolled_by AND password_set
        $stmt = $pdo->prepare("UPDATE students SET password = ?, password_set = 'Yes', status = 'Active', enrolled_by = ? WHERE id = ?");
        $stmt->execute([$passwordHash, $teacherId, $studentId]);

        // 2. Get student's required subjects to create enrollments
        $stmt = $pdo->prepare("SELECT required_subjects, full_name FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if ($student) {
            $requiredSubjects = json_decode($student['required_subjects'] ?: '[]', true);
            if (is_array($requiredSubjects)) {
                foreach ($requiredSubjects as $subject) {
                    // Cek jika sudah ada enrollment (elak duplicate)
                    $stmt = $pdo->prepare("SELECT id FROM student_enrollments WHERE student_id = ? AND subject = ?");
                    $stmt->execute([$studentId, $subject]);
                    if (!$stmt->fetch()) {
                        // Create enrollment with Approved status
                        $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, enrolled_at, approved_at) VALUES (?, ?, ?, 'Approved', NOW(), NOW())");
                        $stmt->execute([$studentId, $teacherId, $subject]);
                    } else {
                        // Update existing enrollment if it was pending
                        $stmt = $pdo->prepare("UPDATE student_enrollments SET teacher_id = ?, status = 'Approved', approved_at = NOW() WHERE student_id = ? AND subject = ?");
                        $stmt->execute([$teacherId, $studentId, $subject]);
                    }
                }
            }

            // Notify student (if we had a notification system for students)
            createNotification($studentId, 'student', 'Permohonan Diluluskan', 'Permohonan anda telah diluluskan oleh Cikgu ' . $_SESSION['full_name'], 'success');
        }

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya diluluskan dan diaktifkan di bawah seliaan anda!']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}

function addStudentManual()
{
    global $pdo;
    requireTeacher(); // Teacher only

    // 1. Get Data
    $fullName = $_POST['full_name'] ?? '';
    $age = $_POST['age'] ?? '';
    $phone = $_POST['phone'] ?? '';
    $parentName = $_POST['parent_guardian_name'] ?? '';
    $educationLevel = $_POST['education_level'] ?? '';
    $gradeLevel = $_POST['grade_level'] ?? '';
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';

    // Subjects array
    $requiredSubjects = isset($_POST['required_subjects']) ? $_POST['required_subjects'] : [];

    // 2. Validation
    if (empty($fullName) || empty($username) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Sila lengkapkan semua maklumat wajib.']);
        return;
    }

    // Check username exists
    $stmt = $pdo->prepare("SELECT id FROM students WHERE username = ?");
    $stmt->execute([$username]);
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Nama pengguna sudah wujud. Sila guna yang lain.']);
        return;
    }

    try {
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);

        // Manual Add = Active Status immediately
        // Also enrolled_by = current teacher (since they added them manually)
        $status = 'Active';
        $enrolledBy = $_SESSION['user_id'];

        $stmt = $pdo->prepare("INSERT INTO students (username, password, full_name, age, parent_guardian_name, phone, education_level, grade_level, required_subjects, status, enrolled_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

        $stmt->execute([
            $username,
            $passwordHash,
            $fullName,
            $age,
            $parentName,
            $phone,
            $educationLevel,
            $gradeLevel,
            json_encode($requiredSubjects),
            $status,
            $enrolledBy
        ]);

        echo json_encode(['success' => true, 'message' => 'Pelajar berjaya ditambah manual!']);

    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat Pangkalan Data: ' . $e->getMessage()]);
    }
}

function assignStudentMentor()
{
    global $pdo;
    requireTeacher();

    $studentId = isset($_POST['student_id']) ? (int) $_POST['student_id'] : 0;

    if ($studentId <= 0) {
        echo json_encode(['success' => false, 'message' => 'ID Pelajar tidak sah']);
        return;
    }

    try {
        $teacherId = $_SESSION['user_id'];

        // Check if student already has a primary mentor
        $stmt = $pdo->prepare("SELECT enrolled_by, full_name FROM students WHERE id = ?");
        $stmt->execute([$studentId]);
        $student = $stmt->fetch();

        if (!$student) {
            echo json_encode(['success' => false, 'message' => 'Pelajar tidak dijumpai']);
            return;
        }

        if (empty($student['enrolled_by'])) {
            // Case 1: No Mentor -> Assign as Primary Mentor
            $stmt = $pdo->prepare("UPDATE students SET enrolled_by = ? WHERE id = ?");
            $stmt->execute([$teacherId, $studentId]);
            $msg = 'Anda kini adalah Mentor Utama untuk ' . $student['full_name'];

        } else {
            // Case 2: Has Mentor -> Add as Additional Teacher (via student_enrollments)

            // Check if already enrolled
            $checkStmt = $pdo->prepare("SELECT id FROM student_enrollments WHERE student_id = ? AND teacher_id = ?");
            $checkStmt->execute([$studentId, $teacherId]);

            if ($checkStmt->rowCount() > 0) {
                // If pending/approved, update status AND subject

                // Determine subject again to be sure
                $stmtT = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
                $stmtT->execute([$teacherId]);
                $teacherObj = $stmtT->fetch();
                $tSubjects = json_decode($teacherObj['subjects'] ?? '[]', true) ?: [];

                $stmtS = $pdo->prepare("SELECT required_subjects FROM students WHERE id = ?");
                $stmtS->execute([$studentId]);
                $studentObj = $stmtS->fetch();
                $sSubjects = json_decode($studentObj['required_subjects'] ?? '[]', true) ?: [];

                $intersect = array_intersect($tSubjects, $sSubjects);
                $enrollSubject = 'General';
                if (!empty($intersect)) {
                    $enrollSubject = array_values($intersect)[0];
                } elseif (!empty($tSubjects)) {
                    $enrollSubject = $tSubjects[0];
                }

                $stmt = $pdo->prepare("UPDATE student_enrollments SET status = 'Approved', subject = ? WHERE student_id = ? AND teacher_id = ?");
                $stmt->execute([$enrollSubject, $studentId, $teacherId]);
                $msg = 'Pelajar ini sudah ada dalam senarai anda. Subjek dikemaskini ke: ' . $enrollSubject;
            } else {
                // Determine subject
                // Fetch teacher's subjects
                $stmtT = $pdo->prepare("SELECT subjects FROM teachers WHERE id = ?");
                $stmtT->execute([$teacherId]);
                $teacherObj = $stmtT->fetch();
                $tSubjects = json_decode($teacherObj['subjects'] ?? '[]', true) ?: [];

                // Fetch student's required subjects
                $stmtS = $pdo->prepare("SELECT required_subjects FROM students WHERE id = ?");
                $stmtS->execute([$studentId]);
                $studentObj = $stmtS->fetch();
                $sSubjects = json_decode($studentObj['required_subjects'] ?? '[]', true) ?: [];

                // Find intersection
                $intersect = array_intersect($tSubjects, $sSubjects);
                $enrollSubject = 'General';

                if (!empty($intersect)) {
                    $enrollSubject = array_values($intersect)[0]; // Pick first matching subject
                } elseif (!empty($tSubjects)) {
                    $enrollSubject = $tSubjects[0]; // Fallback to teacher's first subject
                }

                $stmt = $pdo->prepare("INSERT INTO student_enrollments (student_id, teacher_id, subject, status, approved_at) VALUES (?, ?, ?, 'Approved', NOW())");
                $stmt->execute([$studentId, $teacherId, $enrollSubject]);
                $msg = 'Anda telah ditambah sebagai Guru Tambahan untuk ' . $student['full_name'] . ' (' . $enrollSubject . ')';
            }
        }

        echo json_encode(['success' => true, 'message' => $msg]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Ralat: ' . $e->getMessage()]);
    }
}

// ========== BILIK TUTOR FUNCTIONS ==========

/**
 * Get recent messages from Tutor Room chat
 */
function getTutorMessages()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->query("
            SELECT m.*, t.full_name as teacher_name, t.profile_emoji 
            FROM tutor_messages m
            JOIN teachers t ON m.teacher_id = t.id
            ORDER BY m.created_at DESC
            LIMIT 50
        ");
        $messages = array_reverse($stmt->fetchAll());
        echo json_encode(['success' => true, 'messages' => $messages]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Send a message to Tutor Room chat
 */
function sendTutorMessage()
{
    global $pdo;
    requireTeacher();

    $teacherId = $_SESSION['user_id'];
    $message = isset($_POST['message']) ? trim($_POST['message']) : '';

    if (empty($message)) {
        echo json_encode(['success' => false, 'message' => 'Mesej tidak boleh kosong']);
        return;
    }

    try {
        $stmt = $pdo->prepare("INSERT INTO tutor_messages (teacher_id, message) VALUES (?, ?)");
        $stmt->execute([$teacherId, $message]);
        echo json_encode(['success' => true, 'message' => 'Mesej dihantar']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get list of shared files in Tutor Room
 */
function getTutorFiles()
{
    global $pdo;
    requireTeacher();

    try {
        $stmt = $pdo->query("
            SELECT f.*, t.full_name as teacher_name 
            FROM tutor_files f
            JOIN teachers t ON f.teacher_id = t.id
            ORDER BY f.created_at DESC
        ");
        $files = $stmt->fetchAll();
        echo json_encode(['success' => true, 'files' => $files]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Upload a file to Tutor Room
 */
function uploadTutorFile()
{
    global $pdo;
    requireTeacher();

    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        echo json_encode(['success' => false, 'message' => 'Tiada fail dimuat naik']);
        return;
    }

    $teacherId = $_SESSION['user_id'];
    $file = $_FILES['file'];
    $fileName = basename($file['name']);
    $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

    // Allowed extensions
    $allowed = ['pdf', 'docx', 'doc', 'jpg', 'jpeg', 'png', 'ppt', 'pptx', 'xls', 'xlsx', 'zip'];
    if (!in_array($fileExt, $allowed)) {
        echo json_encode(['success' => false, 'message' => 'Jenis fail tidak dibenarkan']);
        return;
    }

    // New filename to prevent collisions
    $newFileName = time() . '_' . preg_replace("/[^a-zA-Z0-9.]/", "_", $fileName);
    $targetDir = __DIR__ . '/assets/uploads/tutor_files/';
    $targetPath = $targetDir . $newFileName;
    $dbPath = 'assets/uploads/tutor_files/' . $newFileName;

    try {
        if (move_uploaded_file($file['tmp_name'], $targetPath)) {
            $stmt = $pdo->prepare("INSERT INTO tutor_files (teacher_id, file_name, file_path, file_type) VALUES (?, ?, ?, ?)");
            $stmt->execute([$teacherId, $fileName, $dbPath, $fileExt]);
            echo json_encode(['success' => true, 'message' => 'Fail berjaya dimuat naik']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Gagal menyimpan fail ke folder']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Update (rename) a tutor file's display name
 */
function updateTutorFile()
{
    global $pdo;
    requireTeacher();

    $fileId = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $newName = isset($_POST['file_name']) ? trim($_POST['file_name']) : '';
    $teacherId = $_SESSION['user_id'];

    if ($fileId <= 0 || empty($newName)) {
        echo json_encode(['success' => false, 'message' => 'Data tidak lengkap']);
        return;
    }

    try {
        $stmt = $pdo->prepare("UPDATE tutor_files SET file_name = ? WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$newName, $fileId, $teacherId]);

        if ($stmt->rowCount() > 0) {
            echo json_encode(['success' => true, 'message' => 'Nama fail dikemaskini']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Gagal mengemaskini atau anda tiada kebenaran']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Delete a tutor file (both DB and physical)
 */
function deleteTutorFile()
{
    global $pdo;
    requireTeacher();

    $fileId = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $teacherId = $_SESSION['user_id'];

    if ($fileId <= 0) {
        echo json_encode(['success' => false, 'message' => 'ID tidak sah']);
        return;
    }

    try {
        // First get the path to delete the physical file
        $stmt = $pdo->prepare("SELECT file_path FROM tutor_files WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$fileId, $teacherId]);
        $file = $stmt->fetch();

        if (!$file) {
            echo json_encode(['success' => false, 'message' => 'Fail tidak dijumpai atau anda tiada kebenaran']);
            return;
        }

        // Delete recorded path
        $fullPath = __DIR__ . '/' . $file['file_path'];
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }

        // Delete from DB
        $stmt = $pdo->prepare("DELETE FROM tutor_files WHERE id = ? AND teacher_id = ?");
        $stmt->execute([$fileId, $teacherId]);

        echo json_encode(['success' => true, 'message' => 'Fail telah dipadam']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Toggle pin status of a chat message
 */
function togglePinTutorMessage()
{
    global $pdo;
    requireTeacher();

    $messageId = isset($_POST['id']) ? (int) $_POST['id'] : 0;

    if ($messageId <= 0) {
        echo json_encode(['success' => false, 'message' => 'ID mesej tidak sah']);
        return;
    }

    try {
        // Get current status
        $stmt = $pdo->prepare("SELECT is_pinned FROM tutor_messages WHERE id = ?");
        $stmt->execute([$messageId]);
        $msg = $stmt->fetch();

        if (!$msg) {
            echo json_encode(['success' => false, 'message' => 'Mesej tidak dijumpai']);
            return;
        }

        $newStatus = $msg['is_pinned'] ? 0 : 1;

        // If pinning, unpin all others first
        if ($newStatus == 1) {
            $pdo->query("UPDATE tutor_messages SET is_pinned = 0");
        }

        $stmt = $pdo->prepare("UPDATE tutor_messages SET is_pinned = ? WHERE id = ?");
        $stmt->execute([$newStatus, $messageId]);

        echo json_encode(['success' => true, 'message' => $newStatus ? 'Mesej disemat (pinned)' : 'Mesej dinyahsemat']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

/**
 * Get safe list of teachers (names and emojis)
 */
function getTeacherList()
{
    global $pdo;
    if (!isset($_SESSION['user_id'])) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak']);
        return;
    }

    try {
        $stmt = $pdo->query("SELECT id, full_name, profile_emoji, subjects FROM teachers WHERE status = 'Active'");
        $teachers = $stmt->fetchAll();
        echo json_encode(['success' => true, 'teachers' => $teachers]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}

function getModuleStudents()
{
    global $pdo;
    requireTeacher();

    $moduleId = isset($_GET['module_id']) ? (int) $_GET['module_id'] : 0;

    if ($moduleId === 0) {
        echo json_encode(['success' => false, 'message' => 'ID modul tidak valid']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT student_id FROM module_students WHERE module_id = ?");
        $stmt->execute([$moduleId]);
        $students = $stmt->fetchAll(PDO::FETCH_COLUMN);

        echo json_encode(['success' => true, 'student_ids' => $students]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
}


